/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Apr 18 15:33:46 2004 by Jeff Dalton
 * Copyright: (c) 2001, 2004, AIAI, University of Edinburgh
 */

package ix.util.context;

import java.lang.ref.WeakReference;
import java.util.*;

import ix.util.*;

/**
 * A Context is an object that is used as an additional, often
 * implicit, parameter when accessing values, such as in a "get"-method.
 * Instead of a value being stored directly in a field, the field contains
 * a list of associations from contexts to values, arranged so that they
 * can be searched efficiently.  The conventions used in this implementation
 * of contexts are that the associations are held in a chain of
 * {@link ContextLink}s, and the relevant context is obtained from
 * an implementation of {@link ContextHolder}.  So instead of writing
 * something like this in a class definition:
 * 
 * <pre>
 *   ValClass field = new FieldValue();
 *
 *   ValClass getField() {
 *       return field;
 *   }
 *
 *   void setField(ValClass value) {
 *       field = value;
 *   }
 * </pre>
 *
 * you would write something like this:
 *
 * <pre>
 *   ContextLink field = new ContextLink(new FieldValue());
 *
 *   // Here one holder used with all fields, and the holder
 *   // is obtained when this object is created.
 *   ContextHolder contextHolder = Context.getContextHolder();
 *
 *   ValClass getField() {
 *       Context c = contextHolder.getContext();
 *       return (ValClass)Context.getInContext(field, c);
 *   }
 *
 *   void setField(ValClass value) {
 *       Context c = contextHolder.getContext();
 *       Context.setInContext(field, c, value);
 *   }
 * </pre>
 *
 * <p>An alternative is to use objects that hide some of the details.
 * Here is the same example using a {@link ContextValue}:
 *
 * <pre>
 *   // Here one holder used with all fields, and the holder
 *   // is obtained when this object is created.
 *   ContextHolder contextHolder = Context.getContextHolder();
 *
 *   ContextValue field = new ContextValue(contextHolder, new FieldValue());
 *
 *   ValClass getField() {
 *       return (ValClass)field.get();
 *   }
 *
 *   void setField(ValClass value) {
 *       field.set(value);
 *   }
 * </pre>
 *
 * Since the defaut holder used when a ContextValue is constructed is the
 * one returned by <code>Context.getContextHolder()</code>, the example
 * can be simplified further to:
 *
 * <pre>
 *   ContextValue field = new ContextValue(new FieldValue());
 *
 *   ValClass getField() {
 *       return (ValClass)field.get();
 *   }
 *
 *   void setField(ValClass value) {
 *       field.set(value);
 *   }
 * </pre>
 *
 * <p>A context has few properties of its own.  It is simply an object
 * with which the values obtained by context-relative accessors may be
 * associated.  Contexts are arranged in a tree: each context has a
 * single parent and a list of children.  When there is no value directly
 * associated with a context, a value is inherited from the nearest
 * ancestor context that does have a directly associated value.
 * However, when a value is assigned in a context, it is always
 * associated directly with that context, and the associations between
 * ancestor contexts and values are not changed.  This makes it
 * possible to protect the values associated with a context, while
 * still being able to access them, by moving to a descendent context.
 *
 * <p>An efficient search for the value associated with a context, or
 * inherited from an ancestor, is accomplished by exploiting context
 * numbers.  Every context has a number which is assigned when it is
 * created, and newer contexts have higher numbers than older ones.  In
 * particular, a context has a higher number than any of its ancestors.
 * The chain of ContextLinks for a given field has links for 
 * higher-numbered (newer) contexts first.  The algorithm for finding
 * the right value is therefore as follows:
 *
 * <ol>
 * <li>Initialize <i>target</i> to the desired context and <i>chain</i>
 *     to the relevant chain of ContextLinks.
 *
 * <li>Discard entries for contexts newer than <i>target</i> from the
 *     front of <i>chain</i>.  If <i>chain</i> becomes null, then there
 *     is no value for the current context.
 *
 * <li>At this point, the first of the remaining entries in <i>chain</i>
 *     should be for <i>target</i> or for a context older than <i>target</i>.
 *     If it's for <i>target</i>, then we're done: return the associated
 *     value.  Otherwise, set <i>target</i> to the parent of <i>target</i>
 *     and return to step 2.
 * </ol>
 *
 * The getInContext method implements that algorithm.
 *
 * <p>Context-holders are used for two reasons.  First, they make it
 * unnecessary to pass contexts as parameters to "get"- and "set"-methods.
 * This makes "context-layered" objects look much more like ordinary objects.
 * Second, a context-holder may be shared by many objects, so that they all
 * change context at once.
 *
 * <p>However, there is no single way of managing context-holders that
 * is right for every case.  In some applications, a single, global
 * context-holder can be used.  In others, a number of more local
 * context-holders will be needed.
 *
 * <p>These problems are partly addressed by context-holding "strategies".
 * A holding strategy provides a method that returns a context-holder,
 * and the Context class holds a current (global) strategy.  The static
 * method Context.getContextHolder() asks the current strategy for a
 * holder.
 *
 * <p>When using contexts, it is typical to fix the holder used by an
 * object when the object is created (as in the examples above).
 * Otherwise, every access to a context-dependent value would
 * involve asking a strategy for a holder and then asking the holder
 * for its current context, rather than just asking a holder for
 * its current context.
 *
 * @see ContextLink
 * @see #getContextHolder()
 * @see ContextHolder
 * @see ContextHolder#getContext()
 * @see ContextHoldingStrategy
 * @see ContextValue
 * @see LLQueue
 */
public final class Context {

    /* * * Static fields * * */

    // N.B. count and numberToContextTable must be
    // initialized before rootContext.

    /**
     * The number of contexts that have been created.
     */
    static long count = 0;

    /**
     * A mapping from context numbers to (weak references to) contexts.
     */
    static HashMap numberToContextTable = new HashMap();

    /**
     * A context that is an ancestor of all others.
     * The root context is often used for default values. <p>
     *
     * Unlike any other context, the root context's parent is null.
     */
    public static final Context rootContext = new Context();

    /**
     * The current context-holding strategy.
     *
     * @see #getContextHoldingStrategy()
     * @see #setContextHoldingStrategy(ContextHoldingStrategy s)
     * @see #getContextHolder()
     */
    static ContextHoldingStrategy contextHoldingStrategy =
	new GlobalHoldingStrategy();


    /* * * Instance fields * * */

    /**
     * A number, unique to this context and larger than the number
     * of any context created earlier.
     */
    final long number;

    /**
     * This context's parent context.
     */
    final Context parent;

    /**
     * A collection of this context's children.
     */
    final List children = new ArrayList();


    /* * * Constructors * * */

    /**
     * Create a context with the root context as its parent. <p>
     *
     * The constructor is also used once to create the root context
     * itself, and then it's arranged for the root context's parent
     * to be null.
     */
    public Context() {
	this(null);
    }

    /**
     * Create a context with a given parent.  If the parent is null,
     * the root context will be used.
     */
    public Context(Context parent) {
	synchronized (Context.class) {
	    number = count++;
	    if (parent == null && rootContext != null)
		this.parent = rootContext;
	    else
		this.parent = parent;
	    if (this.parent != null)
		this.parent.children.add(this);
	    numberToContextTable.put(new Long(this.number),
				     new WeakReference(this));
	    Debug.expect(getContext(number) == this);
	}
    }


    /* * * Instance methods * * */

    /**
     * Returns this context's number.
     */
    public long getNumber() {
	return number;
    }

    /**
     * Returns this context's parent context.
     */
    public Context getParent() {
	return parent;
    }

    /**
     * Returns an unmodifiable view of this context's children.
     */
    public List getChildren() {
	return Collections.unmodifiableList(children);
    }

    /**
     * Remove this context and all of its descendents from the
     * context tree and the number-to-context table.
     */
    public void discard() {
	synchronized (Context.class) {
	    // Remove this from parent's children
	    if (parent != null) {
		parent.children.remove(this);
	    }
	    // Recursively discard.
	    discardSubtree(this);
	}
    }

    private static void discardSubtree(Context c) {
	// Discard child subtrees.
	for (Iterator i = c.children.iterator(); i.hasNext();) {
	    Context child = (Context)i.next();
	    discardSubtree(child);
	}
	// Forget the children
	c.children.clear();
	// Remove from number-to-context table.
	Long n = new Long(c.number);
	WeakReference r = (WeakReference)numberToContextTable.get(n);
	r.clear();		// need we bother?
	numberToContextTable.remove(n);
	Debug.expect(getContext(c.number) == null);
    }

    public String toString() {
	return "#<Context " + number + ">";
    }


    /*************************
     * Useful static methods *
     *************************/

    /**
     * Returns the current context from the ContextHolder returned
     * by the current ContextHoldingStrategy.
     *
     * @see ContextHolder#getContext()
     */
    public static Context getContext() {
	return contextHoldingStrategy.getContextHolder().getContext();
    }

    /**
     * Returns the context that has the indicated number, or null
     * if that context has been discarded or has ceased to exist.
     */
    public static Context getContext(long n) {
	WeakReference r = (WeakReference)
	    numberToContextTable.get(new Long(n));
	return (r == null) ? null : (Context)r.get();
    }

    /**
     * Sets the context in the ContextHolder returned by the current
     * ContextHoldingStrategy.
     *
     * @see ContextHolder#setContext(Context)
     */
    public static void setContext(Context c) {
	contextHoldingStrategy.getContextHolder().setContext(c);
    }

    /**
     * Removes all contexts except the root context.  After this,
     * the root context will have no children, and the only context
     * accessible from its number will be the root context.  Finally,
     * it calls the {@link ContextHoldingStrategy#clearContexts()}
     * method of the current ContextHoldingStrategy.  The strategy
     * should set the current context of any holders it knows about
     * to the root context.
     */
    public static void clearContexts() {
	synchronized(Context.class) {
	    rootContext.children.clear();
	    numberToContextTable.clear();
	}
	contextHoldingStrategy.clearContexts();
    }

    /**
     * Returns the current context-holding strategy.
     *
     * <p>Initially, the strategy is one that provides a single,
     * global context holder.
     *
     * @see GlobalHoldingStrategy
     */
    public static ContextHoldingStrategy getContextHoldingStrategy() {
	return contextHoldingStrategy;
    }

    /**
     * Sets the current context-holding strategy.
     */
    public static void setContextHoldingStrategy(ContextHoldingStrategy s) {
	contextHoldingStrategy = s;
    }

    /**
     * Returns the context-holder provided by the current context-holding
     * strategy.
     *
     * @see #getContextHoldingStrategy()
     */
    public static ContextHolder getContextHolder() {
	return contextHoldingStrategy.getContextHolder();
    }

    /**
     * Searches a chain of ContextLinks to find the value associated
     * with a given context.
     *
     * @return  an Object or null
     */
    public static Object getInContext(ContextLink cl, Context c) {
	for (Context target = c; target != null; target = target.parent) {
	    // Skip entries for contexts newer than target
	    long target_n = target.number;
	    while (cl != null && cl.contextNumber > target_n) {
		cl = cl.next;
	    }
	    if (cl == null)
		return null;
	    else if (cl.contextNumber == target_n)
		return cl.value;
	    // At this point, all remaining entries are for contexts
	    // older than the target but perhaps not older than the
	    // target's parent.
	}
	return null;
    }

    /**
     * Searches a chain of ContextLinks to find the value associated
     * with a given context and returns a default value if no value
     * can be found.
     */
    public static Object getInContext(ContextLink cl, Context c,
				      Object defaultValue) {
	// Assume that this doesn't have to be maximally efficient
	// and so can affort the extra level of call.
	ContextLink link = getValueLinkInContext(cl, c);
	if (link == null)
	    return defaultValue;
	else
	    return link.value;
    }

    /**
     * Searches a chain of ContextLinks to find the value associated
     * with a given context and throws an exception if no value can be found.
     *
     * @throws IllegalArgumentException if no value can be found
     */
    public static Object requireInContext(ContextLink cl, Context c) {
	// Assume that this doesn't have to be maximally efficient
	// and so can affort the extra level of call.
	ContextLink link = getValueLinkInContext(cl, c);
	if (link == null)
	    throw new IllegalArgumentException
		("Couldn't find value in " + c);
	else
	    return link.value;
    }

    /**
     * Searches a chain of ContextLinks to find the link that
     * contains the value associated with a given context.
     * Note that the link may be for an ancestor of the specified
     * context.
     *
     * @return  a ContextLink or null
     */
    public static ContextLink getValueLinkInContext
	                          (ContextLink cl, Context c) {
	for (Context target = c; target != null; target = target.parent) {
	    // Skip entries for contexts newer than target
	    long target_n = target.number;
	    while (cl != null && cl.contextNumber > target_n) {
		cl = cl.next;
	    }
	    if (cl == null)
		return null;
	    else if (cl.contextNumber == target_n)
		return cl;
	    // At this point, all remaining entries are for contexts
	    // older than the target but perhaps not older than the
	    // target's parent.
	}
	return null;
    }

    /**
     * Modifies a chain of ContextLinks to associate a value directly
     * with a given context.
     */
    public static void setInContext(ContextLink cl, Context c, Object value) {
	Context target = c;
	long target_n = target.number;
	ContextLink at = cl;
	while (true) {
	    if (at.contextNumber == target_n) {
		at.value = value;
		return;
	    }
	    else if (at.contextNumber < target_n) {
		at.next = new ContextLink
		    (at.contextNumber, at.value, at.next);
		at.contextNumber = target_n;
		at.value = value;
		return;
	    }
	    else if (at.next == null) {
		at.next = new ContextLink(target, value, null);
		return;
	    }
	    else {
		at = at.next;
	    }
	}
    }

    /**
     * Temporarily changes a context-holder's current context around
     * a call to a Runnable's run() method.
     */
    public static void inContext(ContextHolder h, Context c, Runnable r) {
	Context saved = h.getContext();
	try {
	    h.setContext(c);
	    r.run();
	}
	finally {
	    h.setContext(saved);
	}
    }

    /**
     * Moves the specified context-holder to a new child of its
     * current context and returns that new context.
     */
    public static Context pushContext(ContextHolder h) {
	h.setContext(new Context(h.getContext()));
	return h.getContext();
    }

    /**
     * Moves the specified context-holder to the parent of its current
     * context.
     */
    public static Context popContext(ContextHolder h) {
	Context c = h.getContext();
	if (c == null)
	    throw new Error("Trying to pop null context");
	else if (c.parent == null)
	    throw new Error("Trying to pop to null context");
	else
	    h.setContext(c.parent);
	return c;
    }

    /**
     * Calls {@link #pushContext(ContextHolder)} on the ContextHolder
     * returned by the current ContextHoldingStrategy.
     */
    public static Context pushContext() {
	return pushContext(contextHoldingStrategy.getContextHolder());
    }

    /**
     * Calls {@link #popContext(ContextHolder)} on the ContextHolder
     * returned by the current ContextHoldingStrategy
     */
    public static Context popContext() {
	return popContext(contextHoldingStrategy.getContextHolder());
    }

    /**
     * Calls {@link #inContext(ContextHolder, Context, Runnable)}
     * on the ContextHolder returned by the current ContextHoldingStrategy.
     */
    public static void inContext(Context c, Runnable r) {
	inContext(contextHoldingStrategy.getContextHolder(), c, r);
    }

    /* * * Debugging utilities * * */

    /**
     * Prints a description of the current context tree on
     * {@link Debug#out}.  Children appear after their parent,
     * indented one more level.
     */
    public static void printContextTree() {
	pctLoop(rootContext, 0);
    }

    private static void pctLoop(Context at, int level) {
	Debug.out.print(Strings.repeat(level, "   "));
	Debug.out.println(at);
	for (Iterator i = at.getChildren().iterator(); i.hasNext();) {
	    Context child = (Context)i.next();
	    pctLoop(child, level + 1);
	}
    }

}

