/* File: Util.java
 * Contains: A class for useful static methods
 * Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Created: January 1998
 * Updated: Thu Sep 21 15:33:26 2006 by Jeff Dalton
 * Copyright: (c) 1998 - 2006, AIAI, University of Edinburgh
 */

package ix.util;

import java.lang.reflect.*;
import java.util.*;
import java.io.*;
import java.net.*;

import java.awt.Component;
import javax.swing.*;

import ix.util.lisp.*;
import ix.util.xml.XML;		// for various utilities

/**
 * Class for useful static methods that don't belong anywhere else.
 *
 * @see Collect
 * @see Fn
 * @see Strings
 */
public class Util {

    /** 
     * A method to print the name of the system, the release version,
     * and the release date.
     */
    public static void printGreeting(String name) {
	System.out.println(name
			   + ", I-X version " + ix.Release.version
			   + ", " + ix.Release.date);
	System.out.println("");
    }

    /** 
     * Asks the GUI event-handling thread to display a text message
     * in a dialogue box and does not return until this has occurred
     * and the user has dismissed the dialogue.
     *
     * <p>If this method is called outside the event dispatching
     * thread, it calls <code>SwingUtilities.invokeAndWait(Runnable)</code>.
     *
     * @param parentComponent determines the Frame in which dialogs
     *        are displayed.
     * @param message the contents of the message.
     */
    public static void displayAndWait(final java.awt.Component parentComponent,
				      final Object message) {
	Runnable run = new Runnable() {
            public void run() {
                JOptionPane.showMessageDialog
		    (parentComponent,
		     message,
		     "Message",
                     JOptionPane.PLAIN_MESSAGE);
            }
        };
	if (SwingUtilities.isEventDispatchThread())
	    run.run();
	else {
	    try {
		SwingUtilities.invokeAndWait(run);
	    }
	    catch (Throwable t) {
	    }
	}
    }

    /**
     * A version of SwingUtilities.invokeAndWait(Runnable) that does
     * not require local handling of checked exceptions.
     *
     * @throws RethrownException if SwingUtilities.invokeAndWait throws
     *   an InvocationTargetException or InterruptedException.
     */
    public static void swingAndWait(Runnable thunk) {
	if (SwingUtilities.isEventDispatchThread()) {
	    thunk.run();
	}
	else {
	    try {
		SwingUtilities.invokeAndWait(thunk);
	    }
	    catch (InvocationTargetException e) {
		Debug.noteln("Invocation target exception", e);
		Debug.noteException(Debug.getExceptionCause(e));
		throw new RethrownException(e);
	    }
	    catch (InterruptedException e) {
		throw new RethrownException(e);
	    }
	}
    }

    /**
     * Brings up a message that the given string item is not yet supported.
     * e.g. "Deleting a schema is not yet supported"
     */
    public static void notImplemented(Component parent, String item){
      JOptionPane.showMessageDialog(parent, 
				    item + " is not yet supported.");
    }

    /**
     * Display a textual input dialog with a specified initial
     * value.
     *
     * <p><i>Such a method is provided by JOptionPane in JDK 1.4
     * and later, but not, it seems, in earlier versions.</i>
     */
    public static String showInputDialog(Component parent, Object message,
					 Object initialValue) {
	return (String)JOptionPane
	    .showInputDialog
	        (parent,
		 message,
		 null,		// title
		 JOptionPane.QUESTION_MESSAGE,
		 null,		// icon
		 null,		// selection values array
		 initialValue);
    }

    /**
     * Displays a confirmation dialog and returns true or false
     * depending on whether the use selected "Yes" or "No".
     * It can be called in any thread and will switch to the
     * Swing / AWT event-dispatching thread to display the dialog
     * if it is called in some other thread.  In all cases,
     * the user must make the selection before this method will
     * return.
     * 
     * @see #swingAndWait(Runnable)
     * @see #displayAndWait(java.awt.Component, Object)
     */
    public static boolean dialogConfirms(final Component parent,
					 final Object text) {
	if (SwingUtilities.isEventDispatchThread()) {
	    return do_dialogConfirms(parent, text);
	}
	else {
	    final boolean[] result = {true}; 	// Gak! /\/
	    swingAndWait(new Runnable() {
		public void run() {
		    result[0] = do_dialogConfirms(parent, text);
		}
	    });
	    return result[0];
	}
    }

    private static boolean do_dialogConfirms(Component parent, Object text) {
	switch(JOptionPane.showConfirmDialog(parent,
	          text, "Confirm", JOptionPane.YES_NO_OPTION)) {
	case JOptionPane.YES_OPTION:
	    return true;
	case JOptionPane.NO_OPTION:
	    return false;
	}
	throw new Error("Confirm dialog failed to return YES or NO");
    }

    /**
     * Sleeps for the specified number of seconds.
     */
    public static void sleepSeconds(int seconds) {
	if (seconds > 0) {
	    int delay = seconds * 1000;
	    Debug.noteln("Sleeping", delay);
	    try { Thread.sleep(delay); }
	    catch (InterruptedException e) {}
	}
    }

    /**
     * Returns a Reader for the specified URL based on the InputStream
     * and content encoding obtained by opening a connection to the URL.
     */
    public static Reader openURLReader(URL url) throws IOException {
	URLConnection conn = url.openConnection();
	InputStream stream = conn.getInputStream();
	String encoding = conn.getContentEncoding();
	if (encoding == null)
	    return new InputStreamReader(stream);
	else 
	    return new InputStreamReader(stream, encoding);
    }

    /**
     * Constructs an ImageIcon from the named (image) resource.
     * The resource can be specified as a file-name, a URL,
     * or as the name of a resource that can be found on the
     * agent's class-path.  If the name does not work as-is,
     * and it is not a URL and does not contain "/", it is
     * tried again with "resources/images/" added to the front.
     *
     * @see XML#toURL(String)
     */
    public static ImageIcon getImageIcon(String name) {
	URL url = XML.toURL(name);
	if (url == null) {
	    // /\/: We know it would not be null if it was given as a URL
	    if (name.indexOf('/') < 0)
		url = XML.toURL("resources/images/" + name);
	}
	if (url != null) {
	    // Debug.noteln("Making ImageIcon from", url);
	    return new ImageIcon(url);
	}
	else {
	    throw new RuntimeException("Can't find image resource named "
				       + Strings.quote(name));
	}
    }

    /**
     * Constructs an ImageIcon from the named (image) resource found
     * on the system's class path.  If the name contains "/", it is
     * used as-is; else "resources/images/" is added to the front.
     */
    public static ImageIcon resourceImageIcon(String name) {
	String fullName = name.indexOf('/') > -1
	    ? name
	    : "resources/images/" + name;
	// URL imageLocation = ClassLoader.getSystemResource(fullName);
//  	URL imageLocation = Util.class.getClassLoader().getResource(fullName);
	URL imageLocation = XML.toURL(fullName);
	if (imageLocation != null) {
	    // Debug.noteln("Making ImageIcon from", imageLocation);
	    return new ImageIcon(imageLocation);
	}
	else {
	    throw new RuntimeException("Can't find image resource named "
				       + Strings.quote(name));
	}
    }

    /**
     * Returns the name of the user who started this JVM.
     */
    public static String getUserName() {
	return System.getProperty("user.name");
    }


    /**
     * Returns the name of the machine that is running this JVM.
     * The name can be given explicitly by specifying a "host"
     * parameter.
     *
     * @throws UnknownHostException if it cannot determine the
     *    host name and no "host" parameter has been specified.
     * @see ix.util.Parameters
     */
    public static String getHostName() throws UnknownHostException {
	return Parameters.haveParameter("host")
	    ? Parameters.getParameter("host")
	    // N.B. the following call might fail.
	    : InetAddress.getLocalHost().getCanonicalHostName();
    }

    /**
     * Puts double quotes around a string.  This is useful to indicate
     * the boundaries of the string when it's included in other text
     * such as in an error or warning message.
     */
    public static String quote(String text) {
	return "\"" + text + "\"";
    }

    /**
     * A method that lets you write "run" before the creation of
     * a Runnable.  For example:
     * <pre>
     *    Util.run(new Runnable() {
     *        public void run() {
     *            ...
     *       }
     *    });
     * </pre>
     * instead of the more awkward
     * <pre>
     *    new Runnable() {
     *        public void run() {
     *            ...
     *       }
     *    }.run();
     * </pre>
     */
    public static void run(Runnable thunk) {
	thunk.run();
    }

    /**
     * Runs each item, in sequence, in a list of Runnables.
     * "When" is a brief description of when this happens,
     * for example "startup" or "exit".
     */
    public static void runHooks(String when, List hooks) {
        Debug.noteln("Running " + when + " hooks");
        for (Iterator i = hooks.iterator(); i.hasNext();) {
            Runnable hook = (Runnable)i.next();
	    Debug.noteln("Running " + when + " hook", hook);
            // /\/: Wrap a try-catch around it?
	    hook.run();
        }
    }

    /**
     * Returns the class of the specified name or else null if
     * the class cannot be found.  It does this by calling
     * <code>Class.forName(name)</code> and catching any
     * ClassNotFoundException.
     */
    public static Class classForNameElseNull(String name) {
	try {
	    return Class.forName(name);
	}
	catch (ClassNotFoundException e) {
	    return null;
	}
    }

    /**
     * Makes an instance of a class using the 0-argument constructor.
     *
     * @throws RethrownException if the attempt fails, with the
     *    original exception inside it.
     */
    public static Object makeInstance(Class c) {
	try {
	    return c.newInstance();
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new RethrownException
		(e, "Cannot make 0-arg instance of " + c + " because " +
		    Debug.describeException(e));
	}
    }

    /**
     * Makes an instance of a class using a 1-argument constructor.
     * The method first tries to get a constructor whose signature
     * matches the argument's class, then tries again with successive
     * superclasses of the argument's class.
     *
     * @param c    the class to instantiate
     * @param arg  an object to pass to the constructor
     *
     * @throws RethrownException if the attempt fails, with the
     *    original exception inside it.
     */
    public static Object makeInstance(Class c, Object arg) {
	Class[] sig = new Class[1];
	Constructor cons;
	try {
	    for (Class argClass = arg.getClass(); argClass != null
		     ; argClass = argClass.getSuperclass()) {
		sig[0] = argClass;
		try {
		    cons = c.getConstructor(sig);
		}
		catch (NoSuchMethodException e) {
		    continue;
		}
		return cons.newInstance(new Object[]{arg});
	    }
	    throw new NoSuchMethodException("Can't find constructor");
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new RethrownException
		(e, "Cannot make an instance of " + c + 
		    " with one " + arg.getClass() + " parameter" +
		    " because " + Debug.describeException(e));
	}
    }

    /**
     * Clones an object without any declared checked exceptions.
     */
    public static Object clone(Object obj) {
	return Fn.apply(obj, "clone", new Object[]{});
    }

    /**
     * Performs a deep-copy clone by serializing then deserialising
     * an object.
     */
    public static Object cloneBySerializing(Object obj) {
	try {
	    ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
	    ObjectOutputStream objOut = new ObjectOutputStream(byteOut);
	    objOut.writeObject(obj);
	    objOut.flush();
	    ByteArrayInputStream byteIn =
		new ByteArrayInputStream(byteOut.toByteArray());
	    objOut.close();
	    byteOut.close();
	    ObjectInputStream objIn = new ObjectInputStream(byteIn);
	    Object result = objIn.readObject();
	    byteIn.close();
	    objIn.close();
	    return result;
	}
	catch (ClassNotFoundException c) {
	    throw new RethrownException(c);
	}
	catch (IOException e) {
	    throw new RethrownException(e);
	}
    }

    /**
     * Returns an object if it is an instance of the specified class,
     * but throws a ClassCastException if it isn't.  This allows us
     * to provide a more infomative error message than if we'd just
     * cast the object to the desired class.
     */
    public static Object mustBe(Class c, Object o) {
	if (c.isInstance(o))
	    return o;
	else if (o == null)
	    throw new ClassCastException
		("Required " + aClass(c) + " but found null");
	else
	    throw new ClassCastException
		("Required " + aClass(c) +
		 " but found " + aClass(o.getClass()) + ": " + o);
    }

    /**
     * Returns a "nice" name for a class, using dash-syntax rather
     * than Java capitalisation, prefixed by "a" or "an" as appropriate.
     */
    public static String aClass(Class c) {
	// /\/: There's a similar method in FileSyntaxManager.
	// /\/: Both methods say "a" for LLists (and say "l-list")
	return Strings.indefinite(XML.nameForClass(c));
    }

    /**
     * Appends the arrays in an array of arrays.
     * The component class of the result array will be
     * the same as that of the 1st of the appended arrays.
     *
     * @throws IllegalArgumentException  if there isn't at least
     *    one array to append or if any of the objects to append
     *    isn't an array.
     */
    public static Object appendArrays(Object[] arrays) {
	if (arrays.length < 1)
	    throw new IllegalArgumentException
		("Attempt to append zero arrays.");
	int size = 0;
	for (int i = 0; i < arrays.length; i++) {
	    size += Array.getLength(arrays[i]);
	}
	Class eltType = arrays[0].getClass().getComponentType();
	Object result = Array.newInstance(eltType, size);
	int ptr = 0;
	for (int i = 0; i < arrays.length; i++) {
	    int len = Array.getLength(arrays[i]);
	    for (int j = 0; j < len; j++) {
		Array.set(result, ptr++, Array.get(arrays[i], j));
	    }
	}
	return result;
    }

    /**
     * Appends two arrays.  The component class of the result array
     * will be the same as that of the 1st of the appended arrays.
     *
     * @throws IllegalArgumentException  if either of the objects
     *    to append isn't an array.
     */
    public static Object appendArrays(Object array1, Object array2) {
	return appendArrays(new Object[]{array1, array2});
    }

    /**
     * Print the elements of a String[] array to System.out as lines.
     */
    public static void printLines(String[] lines) {
	for (int i = 0; i < lines.length; i++) {
	    System.out.println(lines[i]);
	}
    }

    /**
     * Simple, text-based user interaction.  askLine prints a prompt
     * to System.out and returns a String containing the next line
     * from System.in.<p>
     *
     * If askLine blocks when reading, we'd like other threads to be
     * able to run; but that doesn't seem to happen reliably.  Presumably,
     * this is a bug.  In any case, askLine works around the problem by
     * having a loop that checks whether input is available and sleeps
     * for a second if it isn't.
     */
    public static String askLine(String prompt) {
	System.out.print(prompt + " ");
	System.out.flush();
	// /\/: Should we synchronize on System.in?
	try {
	    while (System.in.available() == 0) {
		try { Thread.sleep(1000); }
		catch (InterruptedException e) {}
	    }
	}
	catch (IOException e) {
	    Debug.noteException(e);
	    return "";
	}
	return Util.readLine(System.in);
    }

    /**
     * Prompts for multi-line textual input.
     *
     * @see #askLine(String)
     * @see #readLines()
     */
    public static String askLines(String prompt) {
	System.out.print(prompt + " ");
	System.out.flush();
	return readLines(System.in);
    }

    /**
     * Reads a line from <code>System.in</code>.
     *
     * @see #readLine(InputStream)
     */
    public static String readLine() {
	return readLine(System.in);
    }

    /** 
     * Reads a line from an InputStream and returns it as a String.
     * In Java, we seem to have to write this ourself unless we wrap
     * a special stream (or Reader) around whatever we want to read.
     * Here we provide a static method, because that's easier to
     * mix with other operations.  The only InputStream method
     * called is read().
     */
    public static String readLine(InputStream is) {
	ByteArrayOutputStream line = new ByteArrayOutputStream(80);

    readLoop:
	while (true) {
	    int c;		// a character as a "byte"
	    try { c = is.read(); }
	    catch (IOException e) {
		Debug.warn("IOException reading line from " + is);
		break readLoop;
	    }
	    if (c == '\n') 	// newline
		break readLoop;
	    else if (c == -1)	// end of stream
		break readLoop;
	    else if (c == '\r') // CR
		;		// (ignore)
	    else
		line.write(c);
	}
	return line.toString();
    }

    /**
     * Reads lines from <code>System.in</code>.
     *
     * @see #readLines(InputStream)
     */
    public static String readLines() {
	return readLines(System.in);
    }

    /**
     * Reads lines up to a blank line, then returns all of the
     * lines, except the blank one, concatenated into one string.
     *
     * @see #readLine(InputStream)
     */
    public static String readLines(InputStream in) {
	List lines = new LinkedList();
	while (true) {
	    String line = Util.readLine(in);
	    if (line.equals(""))
		return Strings.joinLines(lines);
	    else
		lines.add(line);
	}
    }

    /**
     * Renames the file to indicate it is an earlier version so that a
     * new version can then be written.  If a file with the desired
     * "backup name" already exists, it is first deleted.  At present,
     * the backup name is formed by replacing any extension/type
     * with <tt>".bak"</tt>.
     */
    public static void renameToBackup(File file) {
	Debug.expect(file.exists(), "No file", file);
	String name = file.getName();
	String[] parts = Strings.breakAtLast(".", name);
	String baseName = parts[0];
	String type = parts[1];
	String bakName = baseName + ".bak";
	File bakFile = new File(file.getParentFile(), bakName);
	if (bakFile.exists()) {
	    Debug.noteln("Deleting existing backup", bakFile);
	    bakFile.delete();
	}
	Debug.noteln("Renameing " + file + " to " + bakFile);
	file.renameTo(bakFile);
    }


}

// Issues:
// * The quote method is the same as the one in Strings and should
//   probably be eliminated; but it is used in many places.
