/* Author: Jeff Dalton
 * Updated: Thu Mar  2 16:47:54 2006 by Jeff Dalton
 * Copyright: (c) 2001, 2004, AIAI, University of Edinburgh
 */

package ix.util;

import javax.swing.*;
import java.awt.Container;
import java.awt.BorderLayout;
import java.awt.event.*;
import java.util.*;

import ix.util.*;
import ix.util.lisp.*;

import ix.iface.util.CatchingActionListener;
import ix.iface.util.TranscriptTextArea;
import ix.iface.util.ToolManager;
import ix.iface.util.ToolController;
import ix.iface.util.IconImage;
import ix.iface.util.ToolFrame;

/**
 * A text area that has its own frame and some buttons.
 */
public class TextAreaFrame extends TranscriptTextArea {

    protected TFrame frame;
    protected List listeners = new LinkedList();

    public TextAreaFrame(String title) {
	this(title, new String[] {});
    }

    public TextAreaFrame(String title, String[] buttons) {
	this(10, 60, title, buttons); // 10 rows, 60 columns
    }

    public TextAreaFrame(int rows, int columns,
			 String title, String[] buttons) {
	// The rows and cols would be right only with a fixed-width font. /\/
	super(rows, columns);
	frame = makeTFrame(title, buttons);
	finishFrame();
    }

    protected void finishFrame() {
	frame.pack();
	frame.setVisible(true);
    }

    public TFrame getFrame() {		// just in case
	return frame;
    }

    protected TFrame makeTFrame(String title, String[] buttons) { // factory
	return new TFrame(title, buttons);
    }

    protected double getEffectiveWidth() {
	double textWidth = super.getEffectiveWidth();
	double frameWidth = frame.getSize().getWidth();
	return Math.min(textWidth, frameWidth);
    }

    static public interface TListener {
	public void buttonPressed(String actionCommand);
    }

    public void addListener(TListener listener) {
	this.listeners.add(listener);
    }

    public void fireButtonPressed(String actionCommand) {
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    TListener listener = (TListener)i.next();
	    listener.buttonPressed(actionCommand);
	}
    }

    public void whenClosed() {
	setVisible(false);
    }

    // Modify some JTextArea methods and provide some frame-like ones.

    public void setText(String text) {
	super.setText(text);
	frame.validate();
    }

    public void append(String text) {
	super.append(text);
	frame.validate();
    }

    public void setTitle(String frameTitle) {
	frame.setTitle(frameTitle);
    }

    public boolean isVisible() {
	return frame.isVisible();
    }

    public void setVisible(boolean b) {
	frame.setVisible(b);
    }

    public void addToolManager() {
	frame.toolManager = new ToolManager();
	frame.getJMenuBar().add(frame.toolManager.getToolsMenu());
    }

    public void addTool(ToolController tc) {
	frame.toolManager.addTool(tc);
    }

    // Our frame class

    public class TFrame extends ToolFrame {

	protected Container contentPane;

	protected ToolManager toolManager;

	protected TFrame(String title, String[] buttons) {
	    super(title);
	    setIconImage(IconImage.getIconImage(this));
	    // setSize(400, 100);
	    setJMenuBar(makeMenuBar());
	    contentPane = getContentPane();
	    // default is equiv to setLayout(new BorderLayout());

	    contentPane.add(new JScrollPane(TextAreaFrame.this),
			    BorderLayout.CENTER);
	    if (buttons.length > 0)
		contentPane.add(makeButtonPanel(buttons),
				BorderLayout.SOUTH);
	    // Make sure a "close" doesn't just leave it running in the
	    // background...
	    setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
	    addWindowListener(new WindowAdapter() {
		public void windowClosing(WindowEvent e) {
		    Debug.noteln(getTitle() + " windowClosing event");
		    whenClosed();
		}
	    });
	}

	protected JMenuBar makeMenuBar() {
	    JMenuBar bar = new JMenuBar();
	    JMenu fileMenu = new JMenu("File");
	    bar.add(fileMenu);
	    fileMenu.add(makeMenuItem("Close"));
	    return bar;
	}

	public JMenuItem makeMenuItem(String text) {
	    // public so subclasses of TextAreaFrame can use it
	    // without having to make their own TFrame subclass.
	    JMenuItem item = new JMenuItem(text);
	    item.addActionListener
		(CatchingActionListener.listener(buttonListener));
	    return item;
	}

	protected JPanel makeButtonPanel(String[] buttons) {
	    JPanel panel = new JPanel();
	    // Stick with default flowlayout for now.
	    for (int i = 0; i < buttons.length; i++) {
		String actionCommand = buttons[i];
		panel.add(makeButton(actionCommand));
	    }
	    return panel;
	}

	protected JButton makeButton(String command) {
	    JButton b = new JButton(command);
	    b.addActionListener
		(CatchingActionListener.listener(buttonListener));
	    return b;
	}

	protected ActionListener buttonListener = new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		String command = e.getActionCommand();
		TextAreaFrame.this.fireButtonPressed(command);
		if (command.equals("Close") || command.equals("Cancel")) {
		    whenClosed();
		}
	    }
	};

    }

}

// Issues:
// * Consider moving this class to package ix.iface.util.

