/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat May 20 03:19:26 2006 by Jeff Dalton
 * Copyright: (c) 1998 - 2006, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

import ix.util.lisp.*;

/**
 * A class containing useful static string methods.
 */
public final class Strings {

    private Strings() {}	// block instantiation

    /**
     * Converts a collection of Strings to a new String array
     * of the same size.
     */
    public static String[] toArray(Collection strings) {
	return (String[])strings.toArray(new String[strings.size()]);
    }

    /**
     * Returns the index of the first occurrence of any of the specified
     * characters, or -1 if none are found.
     *
     * @param chars  the characters to look for
     * @param s      the string to look in
     */
    public static int indexOfAny(String chars, String s) {
	return indexOfAny(chars, 0, s);
    }

    /**
     * Returns the index of the first occurrence of any of the specified
     * characters, or -1 if none are found.
     *
     * @param chars  the characters to look for
     * @param start  index to start looking
     * @param s      the string to look in
     */
    public static int indexOfAny(String chars, int start, String s) {
	for (int i = start, len = s.length(); i < len; i++) {
	    int found = chars.indexOf(s.charAt(i));
	    if (found != -1)
		return i;
	}
	return -1;
    }

    /**
     * breakAtFirst takes a string containing fields separated by
     * a (string) delimiter and returns a two-element string array containing
     * the substring before the first occurrence of the separator and the
     * substring after.  Neither substring contains the delimiter.  If
     * the delimiter does not appear in the string at all, the values
     * are the string and "".
     *
     * @param separator  the delimiter
     * @param s          the string that may contain it
     */
    public static String[] breakAtFirst(String separator, String s) {
	int i = s.indexOf(separator);
	if (i == -1)
	    return new String[]{s, ""};
	else
	    return new String[]{
		s.substring(0, i),
	        s.substring(i + separator.length())
	    };
    }

    /**
     * breakAtLast takes a string containing fields separated by
     * a (string) delimiter and returns a two-element string array containing
     * the substring before the last occurrence of the separator and the
     * substring after.  Neither substring contains the delimiter.  If
     * the delimiter does not appear in the string at all, the values
     * are the string and "".
     *
     * @param separator  the delimiter
     * @param s          the string that may contain it
     */
    public static String[] breakAtLast(String separator, String s) {
	int i = s.lastIndexOf(separator);
	if (i == -1)
	    return new String[]{s, ""};
	else
	    return new String[]{
		s.substring(0, i),
	        s.substring(i + separator.length())
	    };
    }

    /**
     * Returns a list of the substrings delimited by the given
     * separator.  The separator itself does not appear in any
     * element of the list, and if the separator does not occur,
     * the result is a one-element list containing the string as-is.
     *
     * @param separator  the delimiter
     * @param s          the string that may contain it
     */
    public static List breakAt(String separator, String s) {
	int len = s.length();
	if (len == 0)
	    return Lisp.list();
	int i = s.indexOf(separator);
	if (i < 0)
	    return Lisp.list(s);
	LListCollector substrings = new LListCollector();
	int skip = separator.length();
	int start = 0;
	while (i >= 0) {
	    substrings.add(s.substring(start, i));
	    start = i + skip;
	    if (start == len) break; // shortcut
	    i = s.indexOf(separator, start);
	}
	substrings.add(s.substring(start));
	return substrings.contents();
    }

    /**
     * Returns a list of substrings, breaking the string at
     * every occurrence of a separator charater.  The separators
     * are given in a string so that more than one may be specified.
     * None of the separators will appear in any of the result
     * substrings, and if no separator occurs, the result is a
     * one-element list containing the original string as-is.
     *
     * @param separatorChars   the separators
     * @param s                the string to break into parts
     *
     * @see #breakAt(String, String)
     */
    public static List breakAtAny(String separatorChars, String s) {
	int len = s.length();
	if (len == 0)
	    return Lisp.list();
	int i = indexOfAny(separatorChars, s);
	if (i < 0)
	    return Lisp.list(s);
	LListCollector substrings = new LListCollector();
	int start = 0;
	while (i >= 0) {
	    substrings.add(s.substring(start, i));
	    start = i + 1;
	    if (start == len) break; // shortcut
	    i = indexOfAny(separatorChars, start, s);
	}
	substrings.add(s.substring(start));
	return substrings.contents();
    }

    /**
     * Returns a String formed by appending a List of strings
     * with a separator between adjacent elements.  If none of the
     * strings contain the separator, this is the inverse of the
     * operatrion performed by the breakAt method.
     *
     * @param separator   the delimiter that will separate substrings
     * @param substrings  a list of the Strings to join
     */
    public static String joinWith(String separator, List substrings) {
	// Determine result length
	int s_len = separator.length();
	int len = 0;
	for (Iterator i = substrings.iterator(); i.hasNext();) {
	    String substring = (String)i.next();
	    len += substring.length();
	    if (i.hasNext())
		len += s_len;
	}
	// Build result
	SimpleStringBuffer result = new SimpleStringBuffer(len);
	for (Iterator i = substrings.iterator(); i.hasNext();) {
	    String substring = (String)i.next();
	    result.append(substring);
	    if (i.hasNext())
		result.append(separator);
	}
	return result.toString();
    }

    /**
     * Returns a String formed by appending an array of strings
     * with a separator between adjacent elements.
     *
     * @param separator   the delimiter that will separate substrings
     * @param substrings  an array of the Strings to join
     */
    public static String joinWith(String separator, String[] substrings) {
	return joinWith(separator, Arrays.asList(substrings));
    }

    /**
     * Returns a String formed by appending a list of strings
     * with a line-separator between adjacent elements.
     *
     * @param lines  a list of the Strings to join
     */
    public static String joinLines(List lines) {
	return joinWith(System.getProperty("line.separator"), lines);
    }

    /**
     * Returns a String formed by appending an array of strings
     * with a line-separator between adjacent elements.
     *
     * @param lines  an array of the Strings to join
     */
    public static String joinLines(String[] lines) {
	return joinLines(Arrays.asList(lines));
    }

    /**
     * Returns the substring that ends directly before the first
     * occurrence of the separator.  If the separator does not
     * occur, the entire string is returned as-is.
     *
     * @param separator  the delimiter
     * @param s          the string that may contain it
     */
    public static String beforeFirst(String separator, String s) {
	int i = s.indexOf(separator);
	return (i == -1) ? s : s.substring(0, i);
    }

    /**
     * Returns the substring that begins directly after the first
     * occurrence of the separator.  If the separator does not
     * occur, the entire string is returned as-is.
     *
     * @param separator  the delimiter
     * @param s          the string that may contain it
     */
    public static String afterFirst(String separator, String s) {
	int i = s.indexOf(separator);
	return (i == -1) ? s : s.substring(i + separator.length());
    }

    /**
     * Returns the substring that ends directly before the last
     * occurrence of the separator.  If the separator does not
     * occur, the entire string is returned as-is.
     *
     * @param separator  the delimiter
     * @param s          the string that may contain it
     */
    public static String beforeLast(String separator, String s) {
	int i = s.lastIndexOf(separator);
	return (i == -1) ? s : s.substring(0, i);
    }

    /**
     * Returns the substring starting directly after the last
     * occurrence of the separator.  If the separator does not
     * occur, the entire string is returned as-is.
     *
     * @param separator  the delimiter
     * @param s          the string that may contain it
     */
    public static String afterLast(String separator, String s) {
	int i = s.lastIndexOf(separator);
	return (i == -1) ? s : s.substring(i + separator.length());
    }

    /**
     * Returns the first line of a string.
     */
    public static String firstLine(String s) {
	// For some reason, it doesn't work on Windows to use
	// System.getProperty("line.separator") here.  /\/
	return beforeFirst("\n", s);
    }

    /**
     * Returns a List of the lines in a string.
     */
    public static List breakIntoLines(String text) {
	// For some reason, it doesn't work on Windows to
	// use lineSpearator here.  /\/
	return breakAt("\n", text);
    }

    /**
     * Replaces some spaces with line separators to make a long
     * string more readable in contexts where it would have been
     * displayed as a single line.  Any line separators already
     * in the text are retained and are taken into account when
     * calculating line lengths.
     */
    public static String foldLongLine(String text) {
	return foldLongLine(text, 80, null);
    }

    /**
     * Replaces some spaces with line separators to make a long
     * string more readable in contexts where it would have been
     * displayed as a single line.  Any line separators already
     * in the text are retained and are taken into account when
     * calculating line lengths.
     *
     * <p>The prefix, if not null, is put at the front of every
     * resulting line except the first.</p>
     */
    public static String foldLongLine(String text, int maxLen, String prefix) {
	String newline = System.getProperty("line.separator");
	StringBuffer result = new StringBuffer(text.length() + 16);
	List lines = breakIntoLines(text); // in case there are alread newlines
	int len = 0;
	for (Iterator li = lines.iterator(); li.hasNext();) {
	    String line = (String)li.next();
	    List words = Strings.breakAt(" ", line);
	    for (Iterator wi = words.iterator(); wi.hasNext();) {
		String word = (String)wi.next();
		if (len + word.length() > maxLen) {
		    result.append(newline);
		    if (prefix != null) {
			result.append(prefix);
			len = prefix.length();
		    }
		    else len = 0;
		}
		result.append(word).append(" ");
		len += word.length() + 1;
	    }
	    if (li.hasNext()) {
		result.append(newline);
		if (prefix != null) {
		    result.append(prefix);
		    len = prefix.length();
		}
		else len = 0;
	    }
	}
	return result.toString();
    }

    /**
     * Makes a long string more readable by breaking it into lines.
     * This is a variation on {@link #foldLongLine(String)} that
     * returns an array of lines rather than a string containing
     * containing newline sequences.
     */
    public static String[] foldToArray(String text) {
	return toArray(breakIntoLines(foldLongLine(text)));
    }

    /**
     * Returns a string formed by replacing every occurrence
     * of <i>from</i> with <i>to</i> in <i>source</i>.  If any
     * replacements are made, the result is a new string without
     * any modifications to the original.  If, on the other hand,
     * <i>from</i> does not occur, the original string is
     * returned rather than a copy.
     *
     * @param from    the text to replace
     * @param to      the text to replace it with
     * @param source  the text in which to do the replacing
     *
     * @return  a new string if any replacements were needed;
     *          otherwise the original source string.
     */
    public static String replace(String from, String to, String source) {
	int i = source.indexOf(from);
	if (i < 0)
	    return source;
	int diff = to.length() - from.length();
	int slack = diff <= 0 ? 0 : 10*diff;
	StringBuffer result = new StringBuffer(source.length() + slack);
	int skip = from.length();
	int start = 0;
	while (i >= 0) {
	    result.append(source.substring(start, i))
		  .append(to);
	    start = i + skip;
	    i = source.indexOf(from, start);
	}
	result.append(source.substring(start));
	return result.toString();
    }

    /**
     * Returns a copy of <i>source</i> in any character that appears
     * in <i>from</i> has been replaced by the the corresponding char
     * in <i>to</i>.  If any replacements are made, the result is a 
     * new string without any modifications to the original.  Otherwise,
     * the original string is returned rather than a copy.
     *
     * @param from    chars to replace
     * @param to      the chars to replace them with
     * @param source  the text in which to do the replacing
     *
     * @return  a new string if any replacements were needed;
     *          otherwise the original source string.
     */
    public static String replaceChars(String from, String to, String source) {
	int i = indexOfAny(from, source);
	if (i < 0)
	    return source;
	SimpleStringBuffer result = new SimpleStringBuffer(source.length());
	int skip = 1;
	int start = 0;
	while (i >= 0) {
	    char found = source.charAt(i);
	    int pos = from.indexOf(found);
	    char replacement = to.charAt(pos);
	    result.append(source, start, i);
	    result.append(replacement);
	    start = i + skip;
	    i = indexOfAny(from, start, source);
	}
	result.append(source.substring(start));
	return result.toString();
    }

    /**
     * Puts double quotes around a string.  This is useful to indicate
     * the boundaries of the string when it's included in other text
     * such as in an error or warning message.
     */
    public static String quote(String text) {
	return "\"" + text + "\"";
    }

    /**
     * Returns true if the character is an English vowel (a, e, i, o, or u)
     * and false if it is not.
     */
    public static boolean isVowel(char c) {
	switch (c) {
	case 'a': case 'e': case 'i': case 'o': case 'u':
	    return true;
	default:
	    return false;
	}
    }

    /**
     * Returns a string prefixed by "an " or "a " depending on whether
     * or not it begins with an English vowel.
     *
     * @see #isVowel(char)
     */
    public static String indefinite(String noun) {
	return (isVowel(noun.charAt(0)) ? "an " : "a ") + noun;
    }

    /**
     * Returns a disjunction formed from the elements of a collection.
     * The result contains the elements converted to strings,
     * spearated by commans, and with an " or " before the final
     * element.
     */
    public static String disjunction(Collection items) {
	return junction(items, "or");
    }

    /**
     * Returns a conjunction formed from the elements of a collection.
     * The result contains the elements converted to strings,
     * spearated by commans, and with an " and " before the final
     * element.
     */
    public static String conjunction(Collection items) {
	return junction(items, "and");
    }

    private static String junction(Collection col, String connective) {
	List items = col instanceof List ? (List)col : new ArrayList(col);
	for (ListIterator i = items.listIterator(); i.hasNext();) {
	    i.set(i.next().toString());
	}
	int n = items.size();
	switch (n) {
	case 0: return "";
	case 1: return (String)items.get(0);
	case 2: return items.get(0) + " " + connective + " " + items.get(1);
	default:
	    String allButLast = joinWith(", ", items.subList(0, n-1));
	    return allButLast + ", " + connective + " " + items.get(n-1);
	}
    }

    /**
     * Determines whether there are no upper-case characters
     * in a string.
     */
    public static boolean isAllLowerCase(String s) {
	for (int i = 0, len = s.length(); i < len; i++) {
	    if (Character.isUpperCase(s.charAt(i)))
		return false;
	}
	return true;
    }

    /**
     * Returns a copy of the string in which the first character
     * is in upper case and the rest of the string is left as it was.
     */
    public static String capitalize(String s) {
	// return Character.toUpperCase(s.charAt(0)) + s.substring(1);
	SimpleStringBuffer buf = new SimpleStringBuffer(s.length());
	buf.appendCapitalized(s);
	return buf.toString();
    }

    /**
     * Returns a copy of the string in which the first character
     * is in lower case and the rest of the string is left as it was.
     */
    public static String uncapitalize(String s) {
	// return Character.toLowerCase(s.charAt(0)) + s.substring(1);
	SimpleStringBuffer buf = new SimpleStringBuffer(s.length());
	buf.appendUncapitalized(s);
	return buf.toString();
    }

    /**
     * Returns a String made by appending a specified string <tt>count</tt>
     * times.
     */
    public static String repeat(int count, String s) {
	SimpleStringBuffer buf = new SimpleStringBuffer(count * s.length());
	for (int i = 0; i < count; i++)
	    buf.append(s);
	return buf.toString();
    }

    /**
     * Converts a name in which words are separate by dashes
     * into one that uses Java-style capitalisation instead.
     * The first character of the result will be upper-case,
     * as if it were following the convention for class names.
     *
     * @see #javaNameToDashName(String)
     * @see #dashNameToFullJavaName(String)
     */
    public static String dashNameToJavaName(String name) {
	List words = breakAt("-", name);
	SimpleStringBuffer result = new SimpleStringBuffer(name.length());
	for (Iterator i = words.iterator(); i.hasNext();) {
	    String word = (String)i.next();
	    result.appendCapitalized(word);
	}
	return result.toString();
    }

    /**
     * Like dashNameToJavaName but also allows package names,
     * separated by dots, to prefix the class name.  The package
     * names are used unchanged, because we expect them to follow
     * the convention of being all in lower case.
     *
     * <p>The names of inner classes (which contain dollar-signs)
     * are handled by converting the sequence of class names.</p>
     *
     * @see #dashNameToJavaName(String)
     * @see #fullJavaNameToDashName(String)
     */
    public static String dashNameToFullJavaName(String name) {
	if (name.indexOf(".") == -1)
	    return unDash(name);
	else {
	    String[] parts = breakAtLast(".", name);
	    String packageName = parts[0];
	    String className = parts[1];
	    return packageName + "." + unDash(className);
	}
    }

    private static String unDash(String className) {
	if (className.indexOf("$") == -1)
	    return dashNameToJavaName(className);
	else {
	    List parts = breakAt("$", className);
	    for (ListIterator i = parts.listIterator(); i.hasNext();) {
		String name = (String)i.next();
		i.set(dashNameToJavaName(name));
	    }
	    return joinWith("$", parts);
	}
    }

    /**
     * Converts a name that shows word boundaries using Java-style
     * capitalization to a name in which words are (almost) all lower case
     * and separated by dashes.  The "almost" is because in some cases
     * upper case is preserved so that conversion back to a Java name
     * will be correct.  The division into words is performed by a
     * JavaNameWordIterator; the resulting words are converted to all
     * lower case unless they begin with 2 upper-case characters,
     * and then they are left as-is.
     *
     * @see JavaNameWordIterator
     * @see #dashNameToJavaName(String)
     * @see #fullJavaNameToDashName(String)
     */
    public static String javaNameToDashName(String name) {
	StringBuffer result = new StringBuffer(name.length() + 10);
	for (Iterator i = new JavaNameWordIterator(name); i.hasNext();) {
	    String word = (String)i.next();
	    int len = word.length();
	    if (len < 1)
		throw new ConsistencyException("Misparsed", name);
	    else if (!Character.isUpperCase(word.charAt(0)))
		result.append(word);
	    // Now know len >= 1 and 1st char is upper case
	    else if (len >= 2 && Character.isUpperCase(word.charAt(1)))
		// If 2 upper-case in a row, leave word as-is
		result.append(word);
	    // Now know 1st char is upper case and 2nd char isn't.
	    else
		// else convert to lower case.
		result.append(uncapitalize(word));
	    if (i.hasNext())
		result.append("-");
	}
	return result.toString();
    }

    /**
     * Like javaNameToDashName but also allows package names,
     * separated by dots, to prefix the class name.  The package
     * names are used unchanged, because we expect them to follow
     * the convention of being all in lower case.
     *
     * <p>The names of inner classes (which contain dollar-signs)
     * are handled by converting the sequence of class names.</p>
     *
     * @see #javaNameToDashName(String)
     * @see #dashNameToFullJavaName(String)
     */
    public static String fullJavaNameToDashName(String name) {
	if (name.indexOf(".") == -1)
	    return dash(name);
	else {
	    String[] parts = breakAtLast(".", name);
	    String packageName = parts[0];
	    String className = parts[1];
	    return packageName + "." + dash(className);
	}
    }

    private static String dash(String className) {
	if (className.indexOf("$") == -1)
	    return javaNameToDashName(className);
	else {
	    List parts = breakAt("$", className);
	    for (ListIterator i = parts.listIterator(); i.hasNext();) {
		String name = (String)i.next();
		i.set(javaNameToDashName(name));
	    }
	    return joinWith("$", parts);
	}
    }

}
