/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Jun  4 23:08:23 2003 by Jeff Dalton
 * Copyright: (c) 1998 - 2002, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

/**
 * Simple version of StringBuffer without synchronized methods and
 * with no ability to grow beyond its initial capacity.  It can
 * therefore be used when you know the result length or a definite
 * upper bound.
 *
 * <p>It also has a different, and much smaller, set of methods,
 * Note that the append methods do not return the buffer, and that
 * there is a method for appending a substring.
 */
public final class SimpleStringBuffer {

    private char[] chars;
    private int firstFree = 0;		// equiv to length

    /**
     * Creates an empty buffer with the indicated capacity.
     */
    public SimpleStringBuffer(int capacity) {
	chars = new char[capacity];
    }

    /**
     * Returns the number of characters currently in the buffer.
     */
    public int length() {
	return firstFree;
    }

    /**
     * Appends all the characters in a String.
     */
    public void append(String s) {
	int len = s.length();
	s.getChars(0, len, chars, firstFree);
	firstFree += len;
    }

    /**
     * Appends a substring.  The substring contains the characters at
     * beginIndex through to endIndex - 1, and its length is therefore
     * endIndex - beginIndex.
     */
    public void append(String s, int beginIndex, int endIndex) {
	s.getChars(beginIndex, endIndex, chars, firstFree);
	firstFree += endIndex - beginIndex;
    }

    /**
     * Appends the string with its first character converted to upper case.
     */
    public void appendCapitalized(String s) {
	chars[firstFree++] = Character.toUpperCase(s.charAt(0));
	int len = s.length();
	if (len > 1) {
	    s.getChars(1, len, chars, firstFree);
	    firstFree += len - 1;
	}
    }

    /**
     * Appends the string with its first character converted to lower case.
     */
    public void appendUncapitalized(String s) {
	chars[firstFree++] = Character.toLowerCase(s.charAt(0));
	int len = s.length();
	if (len > 1) {
	    s.getChars(1, len, chars, firstFree);
	    firstFree += len - 1;
	}
    }

    /**
     * Appends a single character.
     */
    public void append(char c) {
	chars[firstFree++] = c;
    }

    /**
     * Returns a String containing all the characters that have so far
     * been added to the buffer.
     */
    public String toString() {
	return new String(chars, 0, firstFree);
    }

}
