/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Dec  6 06:05:02 2005 by Jeff Dalton
 * Copyright: (c) 2000, 2004, 2005, AIAI, University of Edinburgh
 */

package ix.util;

import java.io.*;
import java.util.*;

/**
 * An unchecked exception used to wrap other Throwables when they're
 * rethrown.  Usually the wrapped Throwable is a checked exception
 * that should not or could not be handled locally.
 */
public class RethrownException extends RuntimeException {

    Throwable target;

    /**
     * Constructs an exception with a description of the target
     * as its message.
     *
     * @see Debug#describeException(Throwable)
     */
    public RethrownException(Throwable target) {
	this(target, Debug.describeException(target));
    }

    /**
     * Constructs an exception with the specified target and message.
     * The message should be a description of the problem that does
     * not require a separate look at the target exception.
     */
    public RethrownException(Throwable target, String message) {
	super(message);
	this.target = target;
    }

    /**
     * Constructs an exception with the specified target and a message
     * formed by concatenating the prefix, followed by a space, to the
     * result of calling {@link Debug#describeException(Throwable)}
     * on the target.
     */
    public RethrownException(String messagePrefix, Throwable target) {
	this(target, messagePrefix + " " + Debug.describeException(target));
    }

    /**
     * Returns the Throwable that was wrapped in this exception.
     */
    public Throwable getException() {
	return target;
    }

    /**
     * Prints the stack trace of the wrapped exception to System.err.
     */
    public void printStackTrace() {
	printStackTrace(System.err);
    }

    /**
     * Prints the stack trace of the wrapped exception.
     */
    public void printStackTrace(PrintStream s) {
	s.println(this);
	s.println("");
	s.println("Caused by:");
	s.println("");
	target.printStackTrace(s);
    }

    /**
     * Prints the stack trace of the wrapped exception.
     */
    public void printStackTrace(PrintWriter w) {
	w.println(this);
	w.println("");
	w.println("Caused by:");
	w.println("");
	target.printStackTrace(w);
    }

}
