/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar 18 17:39:07 2004 by Jeff Dalton
 * Copyright: (c) 2002 - 2004, AIAI, University of Edinburgh
 */

package ix.util;

import java.lang.reflect.*;
import java.util.*;

import ix.util.reflect.*;
import ix.util.lisp.*;

import ix.util.xml.XML;
import ix.util.xml.LiteralDocument;

/**
 * Walks objects.
 */
public class ObjectWalker implements ObjectVisitor {

    protected ClassSyntax syntax;

    /**
     * Creates a walker that uses the default ClassSyntax.
     *
     * @see ix.util.xml.XML#config()
     * @see ix.util.xml.XMLConfig#defaultClassSyntax()
     */
    public ObjectWalker() {
	this(XML.config().defaultClassSyntax());
    }

    /**
     * Creates a walker that uses the specified ClassSyntax as a
     * source for ClassDescrs.
     */
    public ObjectWalker(ClassSyntax syntax) {
	this.syntax = syntax;
    }

    /**
     * Returns the first subobject that satisfies a predicate
     */
    public static Object findIf(Object source, final Predicate1 p) {
	ObjectWalker walker = new ObjectWalker() {
	    public void walk(Object o) {
		if (p.trueOf(o))
		    throw new ThrownResult(o);
		super.walk(o);
	    }
	};
	try {
	    walker.walk(source);
	}
	catch (ThrownResult r) {
	    return r.getResult();
	}
	return null;
    }

    /**
     * Uses a HashSet to collect subobjects that satisfy a predicate.
     * Calls {@link #collectIf(Object, Collection, Predicate1)}.
     */
    public static Collection collectIf(Object source, Predicate1 p) {
	return collectIf(source, new HashSet(), p);
    }

    /**
     * Uses the specified result collection to collect subobjects
     * that satisfy a predicate.  An object that satisfies the predicate
     * is added to the result before its subobjects are visited.
     */
    public static Collection collectIf(Object source, 
				       final Collection result,
				       final Predicate1 p) {
	ObjectWalker walker = new ObjectWalker() {
	    public void walk(Object o) {
		if (p.trueOf(o))
		    result.add(o);
		super.walk(o);
	    }
	};
	walker.walk(source);
	return result;
    }

    /**
     * Visits the subobjects of an object.  Whether this is deep
     * (recursive) or shallow (top-level only) depends on the
     * {@link #visitElement(Object)} method.
     */
    public void walk(Object obj) {
	if (obj == null)
	    return;
	Class c = obj.getClass();
	ClassDescr cd = syntax.getClassDescr(c);
	cd.visitObject(obj, this);
    }

    // /\/: The "visit" methods should really be protected,
    // but they have to be public because they're in an interface.

    /**
     * Called on subobjects of the object being walked.
     * The method in the ObjectWalker class calls
     * {@link #walk(Object)} on the element and thus
     * implements a recursive walk.
     */
    public void visitElement(Object elt) {
	walk(elt);
    }

    /**
     * Visits a "primitive" object.  It's status as a primitive was
     * determined by the ClassDescr.  In the ObjectWalker class,
     * visitPrimitive does nothing.
     */
    public void visitPrimitive(Object obj, ClassDescr cd) {
	;
    }

    /**
     * Walks a structure, calling {@link #visitElement(Object)} on
     * each field value.
     */
    public void visitStruct(Object obj, ClassDescr cd) {
	List fields = cd.getFieldDescrs();
	for (Iterator fi = fields.iterator(); fi.hasNext();) {
	    FieldDescr fd = (FieldDescr)fi.next();
	    Object val;
	    try { val = fd.getValue(obj); }
	    catch (Exception e) {
		// /\/: fd.getValue can throw too many things.
		Debug.noteException(e);
		throw new RethrownException(e);
	    }
	    visitElement(val);
	}
    }

    /**
     * Walks embedded XML.  The method in the ObjectWalker class
     * does nothing.
     */
    public void visitXML(LiteralDocument xml, ClassDescr cd) {
	;
    }

    /**
     * Walks a List, calling {@link #visitElement(Object)} on each
     * element of the list.
     */
    public void visitList(List obj, ClassDescr cd) {
	for (Iterator i = obj.iterator(); i.hasNext();) {
	    visitElement(i.next());
	}
    }

    /**
     * Walks a Set, calling {@link #visitElement(Object)} on each
     * member of the set.
     */
    public void visitSet(Set obj, ClassDescr cd) {
	for (Iterator i = obj.iterator(); i.hasNext();) {
	    visitElement(i.next());
	}
    }

    /**
     * Walks a Map, calling {@link #visitElement(Object)} on each
     * key and value.
     */
    public void visitMap(Map obj, ClassDescr cd) {
	for (Iterator i = obj.entrySet().iterator(); i.hasNext();) {
	    Map.Entry entry = (Map.Entry)i.next();
	    visitElement(entry.getKey());
	    visitElement(entry.getValue());
	}
    }

    /** Test loop */
    public static void main(String[] argv) {
	ObjectWalker walker = new ObjectWalker() {
	    public void visitElement(Object obj) {
		Debug.noteln("Visiting", obj);
		walk(obj);
	    }
	};
	for (;;) {
	    String in = Util.askLine("Object:");
	    if (in.equals("bye"))
		return;
	    Object obj = Lisp.readFromString(in);
	    walker.walk(obj);
	}
    }

}
