/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar 18 17:38:48 2004 by Jeff Dalton
 * Copyright: (c) 2002, 2003 AIAI, University of Edinburgh
 */

package ix.util;

import java.lang.reflect.*;
import java.util.*;

import ix.util.reflect.*;
import ix.util.lisp.*;

import ix.util.xml.XML;
import ix.util.xml.LiteralDocument;


/**
 * Copies Objects.
 */
public class ObjectCopier implements ObjectMapper {

    protected ClassSyntax syntax;

    /**
     * Creates a copier that uses the default ClassSyntax.
     *
     * @see ix.util.xml.XML#config()
     * @see ix.util.xml.XMLConfig#defaultClassSyntax()
     */
    public ObjectCopier() {
	this(XML.config().defaultClassSyntax());
    }

    /**
     * Creates a copier that uses the specified ClassSyntax as a
     * source for ClassDescrs.
     */
    public ObjectCopier(ClassSyntax syntax) {
	this.syntax = syntax;
    }

    /**
     * Returns a copy of the object.  Whether this is a deep or shallow
     * (top-level only) copy depends on the {@link #mapElement(Object)}
     * method.
     */
    public Object copy(Object obj) {
	if (obj == null)
	    return null;
	Class c = obj.getClass();
	ClassDescr cd = syntax.getClassDescr(c);
	return cd.mapObject(obj, this);
    }

    /**
     * Called on subobjects of the object being copied to determine
     * what object takes the subobject's place in the copy.  The method
     * in the ObjectCopier class is the identity function.  In a subclass
     * that made a deep (recursive) copy, it would call {@link #copy(Object)}
     * on the element.
     */
    public Object mapElement(Object elt) {
	return elt;
    }

    /**
     * Copies a "primitive" object.  It's status as a primitive was
     * determined by the ClassDescr.  In the ObjectCopier class,
     * mapPrimitive is the identity function.
     */
    public Object mapPrimitive(Object obj, ClassDescr cd) {
	return obj;
    }

    /**
     * Copies a structure, calling {@link #mapElement(Object)} on
     * each field value.
     */
    public Object mapStruct(Object obj, ClassDescr cd) {
	Object copy = Util.makeInstance(cd.theClass);
	List fields = cd.getFieldDescrs();
	try {
	    for (Iterator fi = fields.iterator(); fi.hasNext();) {
		FieldDescr fd = (FieldDescr)fi.next();
		fd.setValue(copy, mapElement(fd.getValue(obj)));
	    }
	    return copy;
	}
	catch (Exception e) {
	    Debug.noteException(e);
	    throw new RethrownException
		(e, "Can't copy struct because " +
		    Debug.describeException(e));
	}
    }

    /**
     * Copies embedded XML.  The method in the ObjectCopier class
     * is the identity function.
     */
    public Object mapXML(LiteralDocument xml, ClassDescr cd) {
	return xml;
    }

    /**
     * Copies a List, calling {@link #mapElement(Object)} on each
     * element of the list.  If the list is an LList, this method calls
     * {@link #copyLList(LList)} to do the work.
     */
    public Object mapList(List obj, ClassDescr cd) {
	if (LList.class.isInstance(obj))
	    return copyLList((LList)obj);
	List copy = (List)Util.makeInstance(obj.getClass());
	for (Iterator i = obj.iterator(); i.hasNext();) {
	    copy.add(mapElement(i.next()));
	}
	return copy;
    }

    /**
     * Copies an LList that has a spine made of instances of any
     * mixture of Cons subclasses, calling {@link #mapElement(Object)}
     * on each list element as it goes.
     *
     * @see ix.util.lisp.LList
     * @see ix.util.lisp.Cons
     */
    public LList copyLList(LList obj) {
	// This is tricky becuase the Cons (sub)Class might change
	// as we go through the list.
	if (obj == Lisp.NIL)
	    return Lisp.NIL;
	Cons copy = Cons.typedCons(obj.getClass(), null, Lisp.NIL);
	Cons ctail = copy;
	Cons at = (Cons)obj;
	for (;;) {
	    Object elt = mapElement(at.car());
	    ctail.setCar(elt);
	    LList next_at = at.cdr();
	    if (next_at == Lisp.NIL)
		return copy;
	    Cons cnext = Cons.typedCons(next_at.getClass(), null, Lisp.NIL);
	    ctail.setCdr(cnext);
	    ctail = cnext;
	    at = (Cons)next_at;
	}
    }

    /**
     * Copies a Set, calling {@link #mapElement(Object)} on each
     * member of the set.
     */
    public Object mapSet(Set obj, ClassDescr cd) {
	Set copy = (Set)Util.makeInstance(obj.getClass());
	for (Iterator i = obj.iterator(); i.hasNext();) {
	    copy.add(mapElement(i.next()));
	}
	return copy;
    }

    /**
     * Copies a Map, calling {@link #mapElement(Object)} on each
     * key and value.
     */
    public Object mapMap(Map obj, ClassDescr cd) {
	Map copy = (Map)Util.makeInstance(obj.getClass());
	for (Iterator i = obj.entrySet().iterator(); i.hasNext();) {
	    Map.Entry entry = (Map.Entry)i.next();
	    copy.put(mapElement(entry.getKey()),
		     mapElement(entry.getValue()));
	}
	return copy;
    }

    /** Test loop */
    public static void main(String[] argv) {
	ObjectCopier copier = new ObjectCopier() {
	    public Object mapElement(Object obj) {
		Debug.noteln("Mapping", obj);
		return obj;
	    }
	};
	for (;;) {
	    String in = Util.askLine("Object:");
	    if (in.equals("bye"))
		return;
	    Object obj = Lisp.readFromString(in);
	    Debug.noteln("Copy:", copier.copy(obj));
	}
    }

}
