/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat May 31 23:11:09 2003 by Jeff Dalton
 * Copyright: (c) 2003 AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

import ix.util.lisp.*;

/**
 * Provides a total ordering for arbitrary objects.
 * It should be consistent with equals.
 */
public class ObjectComparator implements Comparator {

    public ObjectComparator() { }

    /**
     * Returns an int < 0, == 0, or > 0 depending on whether the
     * first object is less than, equal to, or greater than the second.
     *
     * @throws ClassCastException  if the objects cannot be compared.
     */
    public int compare(Object a, Object b) {
	if (a == b)
	    return 0;
	// Try to find an appropriate comparison.
	if (a instanceof Symbol || b instanceof Symbol) {
	    return compareSymbol(a, b);
	}
	else if (a instanceof String && b instanceof String) {
	    return ((String)a).compareTo((String)b);
	}
	else if (a instanceof Number && b instanceof Number) {
	    return compareNumbers((Number)a, (Number)b);
	}
	else if (a instanceof List && b instanceof List) {
	    return compareLists((List)a, (List)b);
	}
	else if (a instanceof Comparable && b instanceof Comparable) {
	    return compareComparable((Comparable)a, (Comparable)b);
	}
	else {
	    return defaultCompare(a, b);
	}
    }

    protected int compareSymbol(Object a, Object b) {
	// At least one of a or b is a Symbol.
	if (a instanceof Symbol) {
	    if (b instanceof Symbol) {
		// Symbol - Symbol
		return a.toString().compareTo(b.toString());
	    }
	    else if (b instanceof String) {
		// Symbol - String
		// If they're equal as strings, consider the symbol as <,
		// else return the string comparison.
		int c = a.toString().compareTo((String)b);
		return c == 0 ? -1 : c;
	    }
	    else {
		// Symbol - ?
		return defaultCompare(a, b);
	    }
	}
	else {
	    // ? - Symbol
	    Debug.expect(b instanceof Symbol, "not a Symbol", b);
	    return -compareSymbol(b, a);
	}
    }

    protected int compareNumbers(Number a, Number b) {
	// /\/: Questionable, but it will do for now.
	// /\/: Not consistent with equals, because some Numbers
	// with different values will map to the same double.
	double da = a.doubleValue();
	double db = b.doubleValue();
	if (da < db)
	    return -1;
	else if (da > db)
	    return 1;
	else
	    return 0;
    }

    protected int compareLists(List a, List b) {
	Iterator i = a.iterator(), j = b.iterator();
	while (true) {
	    if (i.hasNext()) {
		if (j.hasNext()) {
		    // both have another element
		    int c = compare(i.next(), j.next());
		    if (c != 0)
			return c;
		    else
			continue;
		}
		else {
		    // a has another element but b doesn't
		    return 1;
		}
	    }
	    else {
		// a is out of elements
		if (j.hasNext()) {
		    // but b has some more
		    return -1;
		}
		else {
		    // both are out of elements
		    return 0;
		}
	    }
	}
    }

    protected int compareComparable(Comparable a, Comparable b) {
	try {
	    if (a.getClass().isInstance(b))
		return a.compareTo(b);
	    else if (b.getClass().isInstance(a))
		return -(b.compareTo(a));
	}
	catch (ClassCastException e) {
	    Debug.noteException(e);
	}
	return defaultCompare(a, b);
    }

    protected int defaultCompare(Object a, Object b) {
	// /\/: Not consistent with equals, because different
	// objects might turn into the same string.
	return Lisp.printToString(a).compareTo(Lisp.printToString(b));
    }

}
