/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Apr  4 15:59:50 2002 by Jeff Dalton
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

/**
 * An iterator that returns successive words from a name that
 * follows Java capitalization conventions.
 *
 * <p>The words are substrings of the name such that concatenating
 * the words in order would result in a string equal to the original.
 *
 * <p>The words are normally the longest substrings that contain an
 * an initial uppercase character followed by one or more non-uppercase
 * characters.  However, any sequence of uppercase characters before
 * such a word (or reaching the end of name) is also taken as a word.
 * For example:
 * <pre>
 *    NameWordIterator: Name, Word, Iterator
 *    XML: XML
 *    XMLParser: XML, Parser
 *    AnXMLParser: An, XML, Parser
 *    SomeMoreXML: Some, More, XML
 * </pre>
 *
 * <p>If the name begins with a non-uppercase character, the first word
 * is everything up to just before the first uppercase char, or else, if
 * no uppercase character occrs, the entire name.
 */
public class JavaNameWordIterator implements Iterator {

    String name;
    int from = 0;
    int at = 0;
    int len;

    public JavaNameWordIterator(String name) {
	this.name = name;
	this.len = name.length();
    }

    public boolean hasNext() {
	return at < len;
    }

    public Object next() {
	from = at;
	char c = name.charAt(at++);
	if (at == len)			// only one char was left
	    return String.valueOf(c); 	// so return it as a string

	if (Character.isUpperCase(c)) {
	    if (Character.isUpperCase(c = name.charAt(at++))) {
		// Two upper case chars in a row
		// Scan for next non-upper, then go back 1
		while (at < len) {
		    if (!Character.isUpperCase(name.charAt(at))) {
			at--;
			break;
		    }
		    at++;
		}
		return name.substring(from, at);		
	    }
	    // Upper followed by lower - end of word is at next upper
	}

	// We've just seen a non-upper char; Scan for next upper.
	Debug.expect(!Character.isUpperCase(c));
	while (at < len) {
	    if (Character.isUpperCase(name.charAt(at)))
		break;
	    at++;
	}
	return name.substring(from, at);
    }

    public void remove() {
	throw new UnsupportedOperationException
	    ("JavaNameWordIterator.remove()");
    }

    /**
     * Test loop.
     */
    public static void main(String[] argv) {
	for (;;) {
	    String in = Util.askLine("Class or field name:");
	    if (in.equals("bye"))
		return;
	    for (Iterator i = new JavaNameWordIterator(in); i.hasNext();) {
		String word = (String)i.next();
		System.out.println("   " + word);
	    }
	}
    }

}
