/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Sep 15 05:20:58 2006 by Jeff Dalton
 * Copyright: (c) 2002, 2003, AIAI, University of Edinburgh
 */

package ix.util;

import java.lang.reflect.*;
import java.util.*;

/**
 * Function utilities
 *
 * @see Collect
 * @see Strings
 */
public final class Fn {

    private Fn() { }	// block instantiation

    /**
     * Calls the indicated method.
     */
    public static Object apply(Object object,
			       String methodName,
			       List arguments) {
	return apply(object, methodName, arguments.toArray());
    }

    /**
     * Calls the indicated method.
     */
    public static Object apply(Object object,
			       String methodName,
			       Object[] arguments) {
	Class c = object.getClass();
	try {
	    Class[] sig = new Class[arguments.length];
	    for (int i = 0; i < sig.length; i++) {
		sig[i] = arguments[i].getClass();
	    }
	    Method meth = c.getMethod(methodName, sig);
	    try {
		return meth.invoke(object, arguments);
	    }
	    catch (IllegalAccessException e) {
		Debug.noteException(e);
		throw new Error("Cannot access method " + methodName +
				" for class " + c);
	    }
	    catch (InvocationTargetException e) {
		Debug.noteException(e);
		throw new RuntimeException
		    ("Exception from method " +
		     methodName + " from class " + c + ": " +
		     Debug.describeException(e.getTargetException()));
	    }
	}
	catch (NoSuchMethodException e) {
	    Debug.noteException(e);
	    throw new Error("Cannot find method " + methodName +
			    " for class " + c);
	}
    }

    /**
     * Calls the indicated method.
     */
    public static Object apply(Object object,
			       Method meth,
			       Object[] arguments) {
	try {
	    return meth.invoke(object, arguments);
	}
	catch (IllegalAccessException e) {
	    Debug.noteException(e);
	    throw new Error("Cannot access method " + meth);
	}
	catch (InvocationTargetException e) {
	    Debug.noteException(e);
	    throw new RuntimeException
		("Exception from method " + meth + ": " +
		 Debug.describeException(e.getTargetException()));
	}
    }

    /**
     * Calls the indicated static method.
     */
    public static Object applyStatic(Class c,
				     String methodName,
				     List arguments) {
	return applyStatic(c, methodName, arguments.toArray());
    }

    /**
     * Calls the indicated static method.
     */
    public static Object applyStatic(Class c,
				     String methodName,
				     Object[] arguments) {
	try {
	    Class[] sig = new Class[arguments.length];
	    for (int i = 0; i < sig.length; i++) {
		sig[i] = arguments[i].getClass();
	    }
	    Method meth = c.getMethod(methodName, sig);
	    try {
		return meth.invoke(null, arguments);
	    }
	    catch (IllegalAccessException e) {
		Debug.noteException(e);
		throw new Error("Cannot access method " + methodName +
				" for class " + c);
	    }
	    catch (InvocationTargetException e) {
		Debug.noteException(e);
		throw new RuntimeException
		    ("Exception from method " +
		     methodName + " from class " + c + ": " +
		     Debug.describeException(e.getTargetException()));
	    }
	}
	catch (NoSuchMethodException e) {
	    Debug.noteException(e);
	    throw new Error("Cannot find method " + methodName +
			    " for class " + c);
	}
    }

    /**
     * Returns the specified method if it exists, else null.
     */
    public static Method getMethod(Object object,
				   String methodName,
				   Object[] arguments) {
	Class c = object.getClass();
	try {
	    Class[] sig = new Class[arguments.length];
	    for (int i = 0; i < sig.length; i++) {
		sig[i] = arguments[i].getClass();
	    }
	    Method meth = c.getMethod(methodName, sig);
	    return meth;
	}
	catch (NoSuchMethodException e) {
	    return null;
	}
    }

    /**
     * Returns the specified method if it exists, else null.
     */
    public static Method getMethod(Object object,
				   String methodName,
				   Class[] sig) {
	// /\/: Goes up the superclass chain but still requires
	// an exact class match.
	Class c = object.getClass();
	try { return c.getMethod(methodName, sig); }
	catch (NoSuchMethodException e) {
	    return null;
	}
    }

    /**
     * Returns a function that calls the 0-argument method specified
     * by a class and method name.
     */
    public static Function1 accessor(final Class c,
				     final String methodName) {
	final Class[] empty = new Class[] {}; // doubles as sig and arglist
	try {
	    final Method meth = c.getMethod(methodName, empty);
	    return new Function1() {
	        public Object funcall(Object a) {
		    try { 
                        // Cast to Object[] to get non-varargs call.
			return meth.invoke(a, (Object[])empty);
		    }
		    catch (IllegalAccessException e) {
			Debug.noteException(e);
			throw new Error("Cannot access method " + methodName +
					" for class " + c);
		    }
		    catch (InvocationTargetException e) {
			Debug.noteException(e);
			throw new RuntimeException
			    ("Exception from method " +
			     methodName + " from class " + c + ": " +
			     Debug.describeException(e.getTargetException()));
		    }
		}
	    };
	}
	catch (NoSuchMethodException e) {
	    Debug.noteException(e);
	    throw new Error("Cannot find method " + methodName +
			    " for class " + c);
	}
    }

    /**
     * Returns a function that always returns the same value.
     */
    public static Function1 always(final Object value) {
	return new Function1() {
	    public Object funcall(Object arg) {
		return value;
	    }
	};
    }

    /**
     * Returns a predicate that returns false when the given
     * predicate returns true, and true when it returns false.
     */
    public static Predicate1 negate(final Predicate1 p) {
	return new Predicate1() {
	    public boolean trueOf(Object a) {
		return !p.trueOf(a);
	    }
	};
    }

    /**
     * Returns a predicate that test whether an object is an
     * instance of the specified class.
     */
    public static Predicate1 isInstanceOf(final Class c) {
	return new Predicate1() {
	    public boolean trueOf(Object a) {
		return c.isInstance(a);
	    }
	};
    }

    /**
     * A predicate that returns true for any object.
     */
    public static final Predicate1 alwaysTrue = new Predicate1() {
	public boolean trueOf(Object a) {
	    return true;
	}
    };

    /**
     * A predicate that returns false for any object.
     */
    public static final Predicate1 alwaysFalse = new Predicate1() {
	public boolean trueOf(Object a) {
	    return false;
	}
    };

    /**
     * The identity function.
     */
    public static final Function1 identity = new Function1() {
	public Object funcall(Object a) {
	    return a;
	}
    };

}

