/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Nov  8 00:18:20 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

import ix.util.lisp.*;

/**
 * Composes Object-to-Collection mappings ("fans").  A "fan" is
 * just a convenient term for a one-to-many mapping: think of
 * each one-to-many as having a fan-like shape.
 *
 * <p>The result is not constructed until requested by a call
 * to the {@link #getResultMap()} method.  The "apply" methods
 * just remember the map or graph and invalidate any existing
 * result.</p>
 *
 * <p>The result is a map made by a call to {@link #makeResultMap()},
 * and the values in that map are collections each made by the
 * {@link #makeValueCollection()} method.  In this class, they
 * construct a {@link StableHashMap} and a HashSet respectively.
 * Subclasses should override those methods if different classes
 * are desired.</p>
 */
public class FanComposer {

    protected Collection keys;

    protected LListCollector allFans = new LListCollector();

    protected Map result = null;

    /**
     * Construct a FanComposer that will build a map that has
     * the specified keys.
     */
    public FanComposer(Collection keys) {
	this.keys = keys;
    }

    public void apply(Map m) {
	allFans.add(m);		// remember the map
	result = null;		// any existing result is invalid
    }

    public void apply(DirectedGraph g) {
	allFans.add(g);		// remember the graph
	result = null;		// any existing result is invalid
    }

    public void apply(Function1 f) {
	allFans.add(f);		// remember the function
	result = null;		// any existing result is invalid
    }

    /**
     * Return the result as a Map.
     */
    public Map getResultMap() {
	if (result == null)
	    computeResult();
	return result;
    }

    private void computeResult() {
	result = makeResultMap();
	for (Iterator i = keys.iterator(); i.hasNext();) {
	    Object key = i.next();
	    Collection values = makeValueCollection();
	    applyFans(key, values, allFans.contents());
	    result.put(key, values);
	}
	// Replace all of the fans applied so far with the composed
	// result, in case more maps or graphs are later applied.
	allFans.clear();
	allFans.add(result);
    }

    private void applyFans(Object key, Collection values, LList fans) {
	// Apply the first fan to the key to get a collection, vs.
	Object fan = fans.car();
	Collection vs = fan instanceof Map
	    ? (Collection)((Map)fan).get(key)
	    : fan instanceof DirectedGraph
	    ? ((DirectedGraph)fan).getSuccessors(key)
	    : (Collection)((Function1)fan).funcall(key);
	// Discard that fan.
	fans = fans.cdr();
	if (fans.isEmpty()) {
	    // If it was the last fan, add vs to values.
	    values.addAll(vs);
	}
	else {
	    // Otherwise, use the elements of vs as keys when
	    // recursively applying the next fan.
	    for (Iterator i = vs.iterator(); i.hasNext();) {
		Object k = i.next();
		applyFans(k, values, fans);
	    }
	}
    }

    protected Map makeResultMap() {
	return new StableHashMap();
    }

    protected Collection makeValueCollection() {
	return new HashSet();
    }

}
