/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Nov  6 02:06:31 2006 by Jeff Dalton
 * Copyright: (c) 2002 - 2006, AIAI, University of Edinburgh
 */

package ix.util;

import java.util.*;

import ix.util.lisp.*;

/**
 * Collection and Map utilities
 *
 * @see Fn
 * @see Strings
 * @see Util
 */
public final class Collect {

    private Collect() { }	// block instantiation

    /*
     * Collection (List and Set) utilities
     */

    /**
     * Returns true iff the collection is either <code>null</code>
     * or empty.
     */
    public static final boolean isEmpty(Collection c) {
	return c == null || c.isEmpty();
    }

    /**
     * Returns the collections's iterator, treating <code>null</code>
     * as an empty list.
     */
    public static final Iterator iterator(Collection c) {
	return c == null ? Lisp.NIL.iterator() : c.iterator();
    }

    /**
     * Ensures that a value is a List object by returning an empty
     * list if the value is <code>null</code>.
     */
    public static List ensureList(List list) {
	// Can't return Collections.EMPTY_LIST, because it's of a
	// special class that Util.makeInstance and hence map can't handle.
	return list == null ? new ArrayList(0) : list;
    }

    /**
     * Returns a random element of the list.
     */
    public static <E> E randomth(List<E> list, Random random) {
	return list.get(random.nextInt(list.size()));
    }

    /**
     * Determines whether a collection contains an element == to
     * an object.
     * 
     * @return true if the collection contains an element == to
     *    the specified item, and false if it does not.
     */
    public static boolean containsEq(Collection col, Object item) {
	for (Iterator i = col.iterator(); i.hasNext();) {
	    if (i.next() == item)
		return true;
	}
	return false;
    }

    /**
     * Returns the first element of the collection such that
     * the predicate is true.
     */
    public static Object findIf(Collection col, Predicate1 p) {
	for (Iterator i = col.iterator(); i.hasNext();) {
	    Object elt = i.next();
	    if (p.trueOf(elt))
		return elt;
	}
	return null;
    }

    /**
     * Replaces the first occurrence of an item in a list, returning
     * the modified list.  If the item does not occur, the list is
     * returned unchanged.
     */
    public static List replaceFirst(Object from, Object to, List source) {
	for (ListIterator i = source.listIterator(); i.hasNext();) {
	    if (i.next() == from) {
		i.set(to);
		break;
	    }
	}
	return source;
    }

    /**
     * Determines whether two collections have the same elements regardless
     * of order.
     */
    public static boolean equalAsSets(Collection a, Collection b) {
	return a.size() == b.size() && a.containsAll(b);
    }

    /**
     * Checks whether two colletions have any elements in common.
     *
     * <p>For some reason, the Java collections have a containsAll
     * method but no containsAny.  This method fills that gap.
     */
    public static boolean haveCommonElements(Collection a, Collection b) {
	for (Iterator i = a.iterator(); i.hasNext();) {
	    if (b.contains(i.next()))
		return true;
	}
	return false;
    }

    /**
     * Returns a new collection of the same class as its first argument
     * and that contains the elements that appear in both collections.
     */
    public static Collection intersection(Collection a, Collection b) {
	Collection result = initResult(a);
	result.addAll(a);	// inefficient /\/
	result.retainAll(b);
	return getResult(result, a);
    }

    /**
     * Returns a new collection of the same class as its first argument
     * and that contains the elements that appear in the first collection
     * plus all elements from the second collection that were not already
     * in the first.
     */
    public static Collection union(Collection a, Collection b) {
	Collection result = initResult(a);
	result.addAll(a);
	for (Iterator i = b.iterator(); i.hasNext();) {
	    Object e = i.next();
	    if (!a.contains(e))
		result.add(e);
	}
	return getResult(result, a);
    }

    /**
     * Returns a new collection of the same class as its first argument
     * and that contains the elements of the first collection that do not
     * appear in the second.
     */
    public static Collection difference(Collection a, Collection b) {
	Collection result = initResult(a);
	result.addAll(a);	// inefficient /\/
	result.removeAll(b);
	return getResult(result, a);
    }

    /**
     * Returns a new list of the same class as its first argument
     * and that contains the elements of the first list followed by
     * those of the second.
     */
    public static List append(List a, List b) {
	List result = (List)initResult(a);
	result.addAll(a);
	result.addAll(b);
	return (List)getResult(result, a);
    }

    /**
     * Returns a same-class copy of a list in which a list of additional
     * elements has been inserted before the first occurrence of a
     * specified target element.  If the target does not occur in the
     * original list, an IllegalArgumentException is thrown.
     *
     * @param target  the item to insert before
     * @param list    the list to insert into
     * @param add     the elements to insert
     */
    public static List insertBeforeFirst(Object target, List list, List add) {
	List result = (List)initResult(list);
	for (Iterator i = list.iterator(); i.hasNext();) {
	    Object item = i.next();
	    if (!target.equals(item)) {
		result.add(item);
	    }
	    else {
		result.addAll(add);
		result.add(item);
		while (i.hasNext()) { result.add(i.next()); }
		return (List)getResult(result, list);
	    }
	}
	throw new IllegalArgumentException
	    ("Target " + target + " did not appear in " + list);
    }

    // Utilities that let us work with LLists without the
    // code getting too messy. /\/

    private static final Collection initResult(Collection a) {
	return a instanceof LList
	    ? new LListCollector()
	    : (Collection)Util.makeInstance(a.getClass());
    }
    private static final Collection getResult(Collection result,
					      Collection a) {
	return a instanceof LList
	    ? ((LListCollector)result).contents()
	    : result;
    }

    /**
     * Returns a new collection of the same class as the original
     * and with corresponding elements obtained by calling the
     * specified function.
     */
    public static Collection map(Collection c, Function1 f) {
	return getResult(map(initResult(c), c, f), c);
    }

    /**
     * Returns the result collection after adding the values returned
     * by a function applied to each element of another collection.
     */
    public static Collection map(Collection result,
				 Collection c,
				 Function1 f) {
	for (Iterator i = c.iterator(); i.hasNext();) {
	    result.add(f.funcall(i.next()));
	}
	return result;
    }

    /**
     * Tests whether a prediate is true of at least one element of
     * a collection.
     */
    public static boolean some(Collection c, Predicate1 p) {
	for (Iterator i = c.iterator(); i.hasNext();) {
	    if (p.trueOf(i.next()))
		return true;
	}
	return false;
    }

    /**
     * Tests whether a predicate is true of every element of a collection.
     */
    public static boolean every(Collection c, Predicate1 p) {
	for (Iterator i = c.iterator(); i.hasNext();) {
	    if (!p.trueOf(i.next()))
		return false;
	}
	return true;
    }

    /**
     * Returns a new collection of the same class as the original
     * and containing only the elements for which the predicate
     * returned true;
     */
    public static Collection filter(Collection c, Predicate1 p) {
	return getResult(filter(initResult(c), c, p), c);
    }

    /**
     * Returns the result collection after adding the elements
     * of another collection for which the predicate returned true;
     */
    public static Collection filter(Collection result,
				    Collection c,
				    Predicate1 p) {
	for (Iterator i = c.iterator(); i.hasNext();) {
	    Object elt = i.next();
	    if (p.trueOf(elt))
		result.add(elt);
	}
	return result;
    }

    /**
     * Returns the <code>toString()</code> representation of the elements
     * of the collection as transformed by the function and separated by
     * commas.
     */
    public static String elementsToString(Collection c, Function1 f) {
	StringBuffer result = new StringBuffer();
	for (Iterator i = c.iterator(); i.hasNext();) {
	    Object elt = i.next();
	    if (f != null)
		elt = f.funcall(elt);
	    result.append(elt.toString());
	    if (i.hasNext())
		result.append(", ");
	}
	return result.toString();
    }

    /**
     * Returns the <code>toString()</code> representation of the elements
     * of the collection, separated by commas.
     */
    public static String elementsToString(Collection c) {
	return elementsToString(c, null);
    }

    /**
     * Performs "triangle iteration" on a list.  For each element
     * in the list, the specified 2-argument procedure is called
     * on that element and each element after it in the list.
     */
    public static void walkTriangle(List list, Proc walker) {
	for (ListIterator i = list.listIterator() 
		 ; i.hasNext();) {
	    Object elt_i = i.next();
	    for (ListIterator j = list.listIterator(i.nextIndex())
		     ; j.hasNext();) {
		Object elt_j = j.next();
		walker.call(elt_i, elt_j);
	    }
	}
    }

    /**
     * Returns the powerset of a List in a nice order.
     * For example, for list <code>[a, b, c]</code>,
     * the result will be:
     * <pre>
     *  [[], [a], [b], [c], [a, b], [a, c], [b, c], [a, b, c]]
     * </pre>
     */
    public static List powerset(List elements) {
	// /\/: There's probably nicer algorithm.
	// The result will contain 2 to the n lists, where
	// n is the number of elements in the original list.
	List result = new ArrayList(1 << elements.size());
	// We want an LList so that we can look at successive tails.
	// We also want to reverse the original list, because that
	// makes earlier elements in the original list appear
	// earlier in the subset lists and puts the subset lists
	// in a nicer order.
	LList elts = Lisp.NIL;
	for (Iterator i = elements.iterator(); i.hasNext();) {
	    elts = new Cons(i.next(), elts);
	}
	powerset1(result, elts, Lisp.NIL);
	Collections.sort(result, new ShortestFirstComparator());
	return result;
    }

    private static void powerset1(List result, LList tail, LList path) {
	if (tail == Lisp.NIL)
	    result.add(path);
	else {
	    Object elt = tail.car();
	    LList rest = tail.cdr();
	    // Subsets that omit elt
	    powerset1(result, rest, path);
	    // Subsets that include elt
	    powerset1(result, rest, new Cons(elt, path));
	}
    }

    /**
     * A Comparator that orders shorter Collections before longer ones.
     */
    public static class ShortestFirstComparator implements Comparator {
	public ShortestFirstComparator() { }
	public int compare(Object o1, Object o2) {
	    Collection c1 = (Collection)o1;
	    Collection c2 = (Collection)o2;
	    return c1.size() - c2.size();
	}
    }


    /*
     * Map utilities
     */

    /**
     * Returns true iff the Map is either <code>null</code>
     * or empty.
     */
    public static final boolean isEmpty(Map m) {
	return m == null || m.isEmpty();
    }

    /**
     * Ensures that a value is a Map object by returning an empty
     * map if the value is <code>null</code>.
     */
    public static Map ensureMap(Map map) {
	return map == null ? Collections.EMPTY_MAP : map;
    }
    
    /**
     * Checks whether two maps are consistent.
     *
     * @return true iff agree on the values of every key they have
     *    in common.
     */
    public static boolean areConsistent(Map m1, Map m2) {
	for (Iterator i = m1.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Object k = e.getKey();
	    Object v = e.getValue();
	    Object v2 = m2.get(k);
	    if (!(v2 == null || v2.equals(v)))
		return false;
	}
	return true;
    }

    /**
     * Returns a function based on the map.  This is useful
     * in conjunction with {@link #map(Collection, Function1)}
     * and similar methods.
     */
    public static Function1 toFunction(final Map m) {
	return new Function1() {
	    public Object funcall(Object a) {
		return m.get(a);
	    }
	};
    }

    /**
     * Adds entries to a map from a collection of keys and a parallel
     * collection of values.
     *
     * @throws IllegalArgumentException if a key is already present
     *    in the Map or appears more than once in the collection of keys.
     * @throws AssertionFailure if the key and value collections are
     *    of different lengths.
     */
    public static Map extendMap(Map m, Collection keys, Collection values) {
	Iterator ki = keys.iterator(), vi = values.iterator();
	while (ki.hasNext() & vi.hasNext()) {
	    Object k = ki.next(), v = vi.next();
	    if (m.get(k) != null)
		throw new IllegalArgumentException("key already present");
	    else
		m.put(k, v);
	}
	Debug.expect(!ki.hasNext() && !vi.hasNext());
	return m;
    }

    /**
     * Returns a map of the same class as the original but
     * with keys and values exchanged.
     */
    public static Map transposeMap(Map m) {
	Map result = (Map)Util.makeInstance(m.getClass());
	for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Object k = e.getKey();
	    Object v = e.getValue();
	    result.put(v, k);
	}
	return result;
    }

    /**
     * Returns a map of the same class as its first argument
     * and that contains the entries from the first map that
     * do not appear in the second.  This method calls
     * {@link #removeAll(Map, Map)} on a copy of the first
     * map.
     */
    public static Map difference(Map a, Map b) {
	Map result = (Map)Util.makeInstance(a.getClass());
	result.putAll(a);
	removeAll(result, b);
	return result;
    }

    /**
     * Removes all entries from a map that appear in a second map.
     * Returns true if any change was made.  Entries count as the
     * same if the second map has an entry for a key and either
     * both values are null or equals regards the values as the same.
     */
    public static boolean removeAll(Map a, Map b) {
	boolean changed = false;
	for (Iterator i = a.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Object k = e.getKey();
	    Object v = e.getValue();
	    Object b_v = b.get(k);
	    if ((v == null && b_v == null && b.containsKey(k))
		|| (b_v != null && v.equals(b_v))) {
	      i.remove();
              changed = true;
	    }
	}
	return changed;
    }

    /**
     * Returns a new map of the same class as the original
     * and containing only the (key, value) entries for which
     * the predicate returned true;
     */
    public static Map filter(Map m, Predicate2 p) {
	return filter((Map)Util.makeInstance(m.getClass()), m, p);
    }

    /**
     * Returns the result map after adding a map's (key, value)
     * entries for which a predicate returns true.
     */
    public static Map filter(Map result, Map m, Predicate2 p) {
	for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Object k = e.getKey();
	    Object v = e.getValue();
	    if (p.trueOf(k, v))
		result.put(k, v);
	}
	return result;
    }

}
// Issues:
// * The methods that work with LLists will use plain Conses
//   in results derived from LListCollectors rather than any
//   Cons subclass used in the arguments.
// * Have a separate Maps class for static map utilities?
