/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Jun 29 16:53:47 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.test;

import java.io.*;
import java.util.*;

import ix.ip2.*;
import ix.iplan.*;
import ix.icore.*;
import ix.icore.process.*;
import ix.icore.plan.Plan;
import ix.icore.domain.*;

import ix.iface.util.KeyValueTable; // for a comparator

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.MatchEnv;

// /\/: This is based on PlanCheckingSimulator and should eventually
// share a common framework.

/**
 * A simple plan-execution simulator that uses refinements for their
 * preconditions and effects.  It does not do expansion into subactivities.
 */
public class SimpleRSim {

    protected Ip2 ip2;

    protected Ip2ModelManager modelManager;

    protected boolean trace = true;

    protected PrintStream traceOut = Debug.out;

    /**
     * Creates a simulator for the specified agent's current plan.
     */
    public SimpleRSim(Ip2 ip2) {
	this.ip2 = ip2;
	this.modelManager = (Ip2ModelManager)ip2.getModelManager();
    }

    /**
     * Creates a simulator for the specified plan and domain.
     */
    public SimpleRSim(Plan plan, Domain domain) {
	this.ip2 = IPlanOptionManager.ModelHolder.newInstance();
	this.modelManager = (Ip2ModelManager)ip2.getModelManager();
	ip2.loadDomain(domain);
	ip2.loadPlan(plan);
    }

    /**
     * Tells this simulator whether or not it should produce trace
     * output.  Trace output shows which node-end is being executed
     * and when conditions and effects are eveluated.
     *
     * @see #setTraceOutput(PrintStream)
     */
    public void setTrace(boolean trace) {
	this.trace = trace;
    }

    /**
     * Sets the stream on which trace output appears.  If this method
     * is not called, the stream {@link Debug#out} will be used.
     *
     * @see #setTrace(boolean)
     */
    public void setTraceOutput(PrintStream ps) {
	this.traceOut = ps;
    }

    public void simulateExecution(List patterns) {
	for (Iterator i = patterns.iterator(); i.hasNext();) {
	    LList pat = (LList)i.next();
	    Activity act = new Activity(pat);
	    try {
		simulateExecution(act);
	    }
	    catch(Exception e) {
		Debug.noteException(e);
	    }
	}
    }

    protected void simulateExecution(Activity act) {
	traceln("Executing", act);
	boolean someMatched = false;
	for (Iterator i = ip2.getDomain().getRefinements().iterator();
	     i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    List envs = modelManager.satisfyRefinementPreconditions(act, r);
	    someMatched = someMatched || envs != null;
	    if (envs != null && !envs.isEmpty()) {
		// Pick one set of bindings.
		MatchEnv env = (MatchEnv)envs.get(0);
		traceln("Using env", env);
		// Fill in variable values in the refinement.
		r = modelManager.fillInRefinement(r, env);
		// Bind any variables that now have bindings,
		// and execute the effects from the refinement;
		List effects = r.getEffectConstraints();
		if (!effects.isEmpty()) {
		    traceln("Effects:", effects);
		}
		modelManager.executeRefinementEffects(r, env);
		return;
	    }
	}
	if (!someMatched)
	    traceln("No refinement matched");
	else
	    traceln("No refinement had satisfiable preconditions");
    }

    class RecordIfUnbound implements Function1 {
	private Set unbound = new TreeSet();
	RecordIfUnbound() {
	};
	public Object funcall(Object name) {
	    unbound.add(Util.mustBe(ItemVar.class, name));
	    return name;
	}
	Set getUnboundVars() {
	    return unbound;
	}
    }

    public void describeFinalWorldState() {
	describeFinalWorldState(traceOut);
    }

    public void describeFinalWorldState(PrintStream out) {
	Map state = new TreeMap(new KeyValueTable.PatternObjectComparator());
	state.putAll(modelManager.getWorldStateMap());
	describeState(out, "Final world state", state);
    }

    protected void describeState(PrintStream out, String label, Map state) {
	if (state.isEmpty())
	    out.println(label + " is empty");
	else {
	    out.println(label + ":");
	    for (Iterator i = state.entrySet().iterator(); i.hasNext();) {
		Map.Entry e = (Map.Entry)i.next();
		out.println("   " + e.getKey() + " = " + e.getValue());
	    }
	}
    }

    /**
     * Topological sort for node-ends to provide a total order
     * that can be used to simulate execution.
     */
    protected class ExecOrderSorter extends TopologicalSorter {
	protected Collection getChildren(Object nodeEnd) {
	    // Return node-ends linked directly after this one.
	    return maybeShuffle(((PNodeEnd)nodeEnd).getSuccessors());
	}
    }

    protected List maybeShuffle(List list) {
	return list;		// never shuffles /\/
    }

    protected List removeIf(final Status status, List nodeEnds) {
	return (List)Collect.filter(nodeEnds, new Predicate1() {
	    public boolean trueOf(Object ne) {
		return ((PNodeEnd)ne).getStatus() != status;
	    }
	});
    }

    protected List keepIf(final Status status, List nodeEnds) {
	return (List)Collect.filter(nodeEnds, new Predicate1() {
	    public boolean trueOf(Object ne) {
		return ((PNodeEnd)ne).getStatus() == status;
	    }
	});
    }

    protected void traceln(String message) {
	if (trace) traceOut.println(message);
    }

    protected void traceln(String message, Object obj) {
	if (trace) traceOut.println(message + " " + obj);
    }

//     protected void doEffects(PNodeEnd ne) {
// 	for (Iterator i = getEffects(ne).iterator(); i.hasNext();) {
// 	    PatternAssignment pv = (PatternAssignment)i.next();
// 	    traceln("  Effect:", pv);
// 	    LList p = (LList)Variable.removeVars(pv.getPattern());
// 	    Object v = Variable.removeVars(pv.getValue());
// 	    worldState.put(p, v);
// 	}
//     }

    /**
     * Standalone main program for testing.  The most useful
     * command-line arguments are:
     * <pre>
     *   -domain=<i>resource name</i>
     *   -trace=true <i>or</i> false<i>, default</i> true
     * </pre>
     */
    public static void main(String[] argv) {

	Debug.off();

	Ip2 ip2 = new ix.test.PlainIp2();
	ip2.mainStartup(argv);	// should load a domain

	// Debug.on = true;

	SimpleRSim sim = new SimpleRSim(ip2);
	sim.processCommandLineArguments();

	while (true)
	    getPatternsAndRun(sim);

    }

    static void getPatternsAndRun(SimpleRSim sim) {

	System.out.println
	    ("\nEnter patterns one per line ending with a blank line.");
	List patterns = new LinkedList();
	while (true) {
	    String line = Util.askLine(">").trim();
	    if (line.equals(""))
		break;
	    patterns.add(Lisp.elementsFromString(line));
	}

	sim.traceln("");
	sim.traceln("- - - - - Beginning simulation - - - - -");
	sim.traceln("");

	sim.simulateExecution(patterns);

	sim.traceln("");
	// sim.report();

	sim.traceln("");
	if (sim.trace) { sim.describeFinalWorldState(); }

	sim.traceln("");
	sim.traceln("- - - - - Simulation ended - - - - -");
	sim.traceln("");

    }

    protected void processCommandLineArguments() {
	setTrace(Parameters.getBoolean("trace", true));
    }

}
