/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Feb 17 19:15:57 2006 by Jeff Dalton
 * Copyright: (c) 2005 - 2006, AIAI, University of Edinburgh
 */

package ix.test;

import java.io.*;
import java.util.*;

import ix.util.*;
import ix.util.xml.*;
import ix.util.reflect.ClassSyntax;

public class SerializationTest {

    XMLConfig config = new ix.ip2.Ip2XMLConfig();
    ClassSyntax syntax = config.defaultClassSyntax();

    StructuralEquality strictEquality = new StrictEquality(syntax);
    StructuralEquality flexibleEquality = new FlexibleEquality(syntax);

    SerializationTest() {
    }

    /**
     * Simple main program used for testing.
     */
    public static void main(String[] argv) {
	new SerializationTest().do_main(argv);
    }

    void do_main(String[] argv) {
	Debug.off();
	Parameters.setIsInteractive(false);
	Parameters.processCommandLineArguments(argv);
	String dirName = Parameters.getParameter("directory", "test-domains");
	// Find all the files in the directory that we can parse.
	FileSyntaxManager fsm = XML.fileSyntaxManager();
	File[] files = fsm.getFiles(Object.class, dirName);
	List problems = new LinkedList();
	for (int i = 0; i < files.length; i++) {
	    try {
		Object obj = XML.readObject(files[i].getPath());
		traceln("File " + files[i].getName() + " contains a " +
			obj.getClass().getName());
		test(obj);
	    }
	    catch (Throwable t) {
		handleException(problems, files[i], t);
	    }
	}
	if (problems.isEmpty()) {
	    traceln("\nNo problems found.");
	    System.exit(0);
	}
	else {
	    traceln("\nProblems:");
	    for (Iterator i = problems.iterator(); i.hasNext();) {
		traceln((String)i.next());
	    }
	    System.exit(1);
	}
    }

    void traceln(String message) {
	System.out.println(message);
    }

    void test(Object obj) {
	Object back = Util.cloneBySerializing(obj);
	Debug.expect(strictEquality.equal(obj, back),
		     "Not structurally equal");
    }

    void handleException(List problems, File f, Throwable t) {
	String message = f.getName() + ":  " + Debug.describeException(t);
	problems.add(message);
	traceln(message);
	if (Debug.on)
	    Debug.displayException(t);
    }

    public static class StrictEquality extends StructuralEquality {

	public StrictEquality(ClassSyntax s) { super(s); }

	// Because the result should always be true, we can
	// throw an imformative exception if it would be false
	// rather than just returning false.  This way we
	// can find out which subobject caused the failure.

	public boolean equal(Object a, Object b) {
	    boolean result = super.equal(a, b);
	    if (result == true)
		return result;
	    else
		throw new IllegalArgumentException
		    ("Mismatched objects " + a + " and " + b);
	}

    }

    public static class FlexibleEquality extends StrictEquality {

	public FlexibleEquality(ClassSyntax s) { super(s); }

	protected boolean haveEquivalentClasses(Object a, Object b) {
	    Class c_a = a.getClass();
	    Class c_b = b.getClass();
	    return c_a == c_b
		|| (a instanceof List && b instanceof List)
		|| (a instanceof Set && b instanceof Set)
		|| (a instanceof Map && b instanceof Map);
	}
    }

}
