/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon May 19 02:34:48 2003 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.test;

import java.util.*;
import java.io.*;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.output.XMLOutputter;

import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.plan.*;
import ix.icore.plan.build.*;
import ix.util.*;
import ix.util.xml.*;
import ix.util.lisp.*;

/**
 * Converts a Compendium map expressed in Compendium's XML syntax
 * to an I-X plan in I-X XML syntax.
 *
 * @see #main(String[])
 */
public class CompendiumToIX {

    /** Compendium node type */
    static final int
	LIST				= 1,
	MAP				= 2,
	ISSUE	/* Question */ 		= 3,
	POSITION /* Answer */		= 4,
	ARGUMENT			= 5,
	PRO				= 6,
	CON				= 7,
	DECISION			= 8,
	REFERENCE			= 9,
	NOTE				= 10,
	
	LIST_SHORTCUT			= 11,
	MAP_SHORTCUT			= 12,
	ISSUE_SHORTCUT			= 13,
	POSITION_SHORTCUT		= 14,
	ARGUMENT_SHORTCUT		= 15,
	PRO_SHORTCUT			= 16,
	CON_SHORTCUT			= 17,
	DECISION_SHORTCUT		= 18,
	REFERENCE_SHORTCUT		= 19,
	NOTE_SHORTCUT			= 20;

    /** Annotation key */
    static final String
	K_ID    = "compendium-id",
	K_LABEL = "compendium-label",
	K_LINKS = "compendium-links";

    PlanBuilder builder = new SimplePlanBuilder();

    Map CompendiumIdToIXObjectMap = new HashMap();

    MultiMap linkMap = new MultiHashMap();

    List issues = new LinkedList();
    List activities = new LinkedList();
    List constraints = new LinkedList();

    public CompendiumToIX() { }

    public Document transform(Document doc) {

	// Convert objects and build useful lists and maps.
	phaseOne(doc);

	// Build the plan.
	phaseTwo();

	// Return the plan.
	Plan plan = builder.getPlan();
	return XML.objectToDocument(plan);
    }

    /**
     * Phase 1 -- convert from Compendium XML to I-X objects.
     */

    void phaseOne(Document doc) {
	Element root = doc.getRootElement();
	List nodes = root.getChild("nodes").getChildren("node");
	List links = root.getChild("links").getChildren("link");

	// Some nodes become I-X objects
	for (Iterator i = nodes.iterator(); i.hasNext();) {
	    Element node = (Element)i.next();
	    Object ixObject = makeIXObject(node);
	    if (ixObject != null) {
		String id = getCompendiumId(node);
		CompendiumIdToIXObjectMap.put(id, ixObject);
	    }
	}

	// Record what's linked to what.
	// /\/: There are link types, but we don't yet know what
	// they mean and so treat them all the same.
	for (Iterator i = links.iterator(); i.hasNext();) {
	    Element link = (Element)i.next();
	    String fromId = link.getAttributeValue("from");
	    String toId = link.getAttributeValue("to");
	    IXObject from = IXObjectForId(fromId);
	    IXObject to = IXObjectForId(toId);
	    if (from != null && to != null) {
		linkMap.addValue(from, new Link(from, to));
	    }
	}
    }

    /**
     * Converts Compendium Elements to I-X objects and adds the
     * I-X objects to any relevant accumulating lists.  Returns
     * null if there is no known conversion.
     */
    Object makeIXObject(Element elt) {
	switch(getCompendiumType(elt)) {
	case ISSUE:		// a.k.a. question
	    Issue issue = makeIXIssue(elt);
	    issues.add(issue);
	    return issue;
	case DECISION:
	    Activity activity = makeIXActivity(elt);
	    activities.add(activity);
	    return activity;
	case ARGUMENT:
	    Constraint con = makeIXConstraint(elt);
	    constraints.add(con);
	    return con;
	default:
	    return null;
	}
    }

    Issue makeIXIssue(Element elt) {
	Issue issue = new Issue(patternFromCompendiumNode(elt));
	addNodeAnnotations(issue, elt);
	return issue;
    }

    Activity makeIXActivity(Element elt) {
	Activity activity = new Activity(patternFromCompendiumNode(elt));
	addNodeAnnotations(activity, elt);
	return activity;
    }

    Constraint makeIXConstraint(Element elt) {
	LList pattern = patternFromCompendiumNode(elt);
	Constraint con =
	    new Constraint("world-state", "effect",
			   Lisp.list(new PatternAssignment(pattern)));
	addNodeAnnotations(con, elt);
	return con;
    }

    LList patternFromCompendiumNode(Element elt) {
	String label = elt.getAttributeValue("label");
	return patternFromText(label == null ? "no pattern" : label);
    }

    /**
     * Converts text taken from a Compendium label to an I-X pattern.
     */
    LList patternFromText(String text) {
	// /\/: Might be tricky.
	try {
	    // Try reading it as symbols, strings, numbers, etc
	    return PatternParser.parse(text);
	}
	catch (Exception e) {
	    Debug.noteln("Trouble reading " + Strings.quote(text) +
			 " because " + Debug.describeException(e));
	    // Make a list of strings by breaking at spaces.
	    return LList.newLList(Strings.breakAt(" ", text));
	}
    }

    /**
     * Give the IXObject any "standard" annotations derived from
     * the Compendium node.  At present, this just records the
     * Compendium id under the key {@link #K_ID}.
     */
    void addNodeAnnotations(IXObject item, Element elt) {
	// Id
	// For now, the Compendium id is used in the link annotation. /\/
	item.setAnnotation(K_ID, getCompendiumId(elt));
    }

    /**
     * Returns the Compenium type as an int, or an int < 0
     * if that type cannot be determined.
     */
    int getCompendiumType(Element elt) {
	String typeValue = elt.getAttributeValue("type");
	if (typeValue == null)
	    return -1;
	try {
	    return Integer.decode(typeValue).intValue();
	}
	catch (NumberFormatException e) {
	    Debug.noteException(e, false);
	    return -2;
	}
    }

    /** Returns the value of an Element's "id" attribute. */
    String getCompendiumId(Element elt) {
	String id = elt.getAttributeValue("id");
	if (id == null || id.equals(""))
	    throw new IllegalArgumentException("Can't get id");
	else
	    return id;
    }

    /**
     * Phase 2 -- build the plan
     */

    void phaseTwo() {

	// Put issues and subissues in the plan.
	for (Iterator i = issues.iterator(); i.hasNext();) {
	    addIssueToPlan((Issue)i.next());
	}

	// Put activities and subactivities in the plan.
	for (Iterator i = activities.iterator(); i.hasNext();) {
	    addActivityToPlan((Activity)i.next());
	}

	// Put constraints in the plan
	for (Iterator i = constraints.iterator(); i.hasNext();) {
	    addConstraintToPlan((Constraint)i.next());
	}

    }

    /**
     * Adds an issue as a subissue of its parent issue, if it
     * has one, else as a top-level issue.
     *
     * @see #findParent(IXObject)
     */
    void addIssueToPlan(Issue issue) {
	Issue parent = (Issue)findParent(issue);
	if (parent == null)
	    builder.addIssue(issue);
	else
	    builder.addSubissue(parent, issue);
	addLinksAnnotation(issue, findParent(issue));
    }

    /**
     * Adds an activity as a subactivity of its parent activity, if it
     * has one, else as a top-level activity.
     *
     * @see #findParent(IXObject)
     */
    void addActivityToPlan(Activity activity) {
	Activity parent = (Activity)findParent(activity);
	if (parent == null)
	    builder.addActivity(activity);
	else
	    builder.addSubactivity(parent, activity);
	addLinksAnnotation(activity, findParent(activity));
    }

    /**
     * Adds a top-level constraint.
     */
    void addConstraintToPlan(Constraint con) {
	// /\/: We add them all at the top level of the plan.
	builder.addConstraint(con);
	addLinksAnnotation(con, null);
    }

    /**
     * Returns the item's parent, if it has one; otherwise,
     * returns null.  The first linked-to object of the
     * same class as the item is taken as the parent.
     */
    IXObject findParent(IXObject item) {
	List linkedOfSameClass = objectsLinkedFrom(item, item.getClass());
	if (linkedOfSameClass.size() > 0)
	    return (IXObject)linkedOfSameClass.get(0); // take 1st as parent
	else
	    return null;
    }

    /**
     * Gives the item an annotation with key {@link #K_LINKS}
     * and value a list of link descriptions.
     */
    void addLinksAnnotation(IXObject item, IXObject parent) {
	List linkInfo = new LinkedList();
	for (Iterator i = getLinks(item).iterator(); i.hasNext();) {
	    Link link = (Link)i.next();
	    IXObject to = (IXObject)link.getTo();
	    if (to != parent) {
		// /\/: Just use a string for now.
		String description =
		    "related-to " + 
		    Strings.afterLast(".", to.getClass().getName()) + " " +
		    to.getAnnotation(K_ID);
		linkInfo.add(description);
	    }
	}
	if (!linkInfo.isEmpty()) {
	    item.setAnnotation(K_LINKS, linkInfo);
	}
    }

    /**
     * Returns the I-X object made from the Compendium node with
     * the given id.
     */
    IXObject IXObjectForId(String id) {
	Debug.expect(id != null, "Null id");
	Debug.expect(!id.equals(""), "Empty id");
	return (IXObject)CompendiumIdToIXObjectMap.get(id);
    }

    /**
     * Returns the list of {@link CompendiumToIX.Link}s
     * that go from the specified {@link IXObject}.
     * Always returns a List, never null.
     */
    List getLinks(IXObject from) {
	List result = (List)linkMap.get(from);
	// Ensure result is not null, and hence that objectsLinkedFrom
	// also never returns null.  /\/
	return result == null ? new LinkedList() : result;
    }

    List objectsLinkedFrom(IXObject from) {
	return objectsLinkedFrom(from, Object.class);
    }

    List objectsLinkedFrom(IXObject from, Class filterCondition) {
	List result = new LinkedList();
	for (Iterator i = getLinks(from).iterator(); i.hasNext();) {
	    Link link = (Link)i.next();
	    Object to = link.getTo();
	    if (filterCondition.isInstance(to))
		result.add(to);
	}
	return result;
    }

    /*
     * Useful classes.
     */

    /** A simple representation of Compendium links. */
    class Link {
	IXObject from;
	IXObject to;
	Link(IXObject from, IXObject to) {
	    this.from = from;
	    this.to = to;
	}
	IXObject getFrom() { return from; }
	IXObject getTo() { return to; }
    }

    public static void main(String[] argv) throws Exception {
	// Turn off debugging output.
	Debug.off();

	if (argv.length < 1) {
	    System.out.println("Args are: file-to-transform");
	    return;
	}

	String filename = argv[0];
	Document doc = XML.parseXML(new File(filename));
	Document result = new CompendiumToIX().transform(doc);
	XML.makePrettyXMLOutputter().output(result, System.out);
	// XML.printXMLWithWhitespace(result, 2);
    }

}
