/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Jan 26 20:06:26 2005 by Jeff Dalton
 * Copyright: (c) 2004, 2005, AIAI, University of Edinburgh
 */

package ix.test;

import java.util.*;

import ix.iplan.*;
import ix.icore.*;
import ix.icore.process.PNodeEnd;
import ix.icore.plan.*;
import ix.icore.plan.build.*;
import ix.icore.domain.*;

import ix.iview.SimpleDomainEditor;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;

/**
 * Used for running block-stacking planning problems.
 */
public class BlockStacker {

    protected Slip slip;
    protected List initialTowers;
    protected List finalTowers;

    // A tower is a sequence of letters, each representing a block,
    // listed from highest block to lowest.  For example, "abc" is a
    // tower with A on top of B on top of C, with C on the table.

    // A block-stacking problem is specified as initial-final
    // where initial and final are comma-separated lists of towers.
    // For example, "ca,b-abc" for the Sussman anomaly.

    public BlockStacker(String problem) {
	setup(problem);
    }

    public static void main(String[] argv) {
	Parameters.processCommandLineArguments(argv);
	String outFile = Parameters.getParameter("output");
	String problem = Parameters.getParameter("problem");
	Debug.expect(problem != null, "No problem specified");
	BlockStacker stacker = new BlockStacker(problem);
	stacker.findPlan();
	while (true) {
	    stacker.checkPlan();
	    if (outFile != null) {
		XML.writeObject(stacker.getPlan(), outFile);
	    }
	    if (Util.askLine("Replan?").trim().startsWith("y"))
		stacker.replan();
	    else
		System.exit(0);
	}
    }

    protected void setup(String problem) {

	// Parse the problem description.
	parseTowers(problem);

	// Create the planner.
	// We make it the main I-X agent but don't give it
	// any command-line arguments.  It will still see any
	// that we process. /\/
	slip = new Slip(true);
	slip.mainStartup(new String[]{});

	// Construct a suitable domain.
	Domain domain = problemDomain(initialTowers, finalTowers);
	String baseDomain = Parameters.getParameter
	    ("base-domain", "domain-library/blocks-pure-htn-1.lsp");
	slip.readDomain(domain, baseDomain);

	XML.writeObject(domain, "/tmp/dom.lsp");

	// Create the initial plan.
	Plan initialPlan = initialPlan(initialTowers, finalTowers);

	// Set up the planner.
	if (Parameters.haveParameter("step-limit"))
	    slip.setStepLimit(Parameters.getInt("step-limit"));
	slip.setDomain(domain);
	slip.loadPlan(initialPlan);

    }

    public void findPlan() {
	slip.plan();
    }

    public void replan() {
	slip.replan();
    }

    public Plan getPlan() {
	return slip.getPlan();
    }

    public void checkPlan() {

	// Simulate plan execution.
	PlanCheckingSimulator sim = new PlanCheckingSimulator(slip);
	sim.run();

	// Print a report that describes any problems found.
	sim.report();

	// Describe the final world state
	sim.describeFinalWorldState();

	// Check that the final state is correct.
	checkTowerState(sim.getWorldStateMap(), finalTowers);

	// Report the key steps in the plan.
	Debug.noteln("Move steps in order:");
	for (Iterator i = sim.getExecutionOrder().iterator(); i.hasNext();) {
	    PNodeEnd ne = (PNodeEnd)i.next();
	    if (ne.getEnd() == End.BEGIN) {
		LList pattern = ne.getNode().getPattern();
		if (pattern.get(0).toString().startsWith("move"))
		    Debug.noteln("   ", pattern);
	    }
	}

    }

    public void parseTowers(String problem) {
	String[] states = Strings.breakAtFirst("-", problem);
	initialTowers = towers(states[0]);
	finalTowers = towers(states[1]);
	Debug.noteln("Initial towers", initialTowers);
	Debug.noteln("Final towers", finalTowers);

	Debug.noteln("First tower's patterns",
		     towerPatterns((List)initialTowers.get(0)));
    }

    /** Returns a list of lists, one list per tower. */
    List towers(String towerDescriptions) {
	List descriptions = Strings.breakAt(",", towerDescriptions);
	List result = new LinkedList();
	for (Iterator i = descriptions.iterator(); i.hasNext();) {
	    result.add(towerBlocks((String)i.next()));
	}
	return result;
    }

    /** Returns a list of symbols representing the blocks in a tower. */
    List towerBlocks(String towerDescription) {
	List result = new LinkedList();
	for (int i = 0, len = towerDescription.length() ; i < len; i++) {
	    String blockName = towerDescription.substring(i, i + 1);
	    result.add(Symbol.intern(blockName));
	}
	return result;
    }

    /*
     * Initial plan
     */

    public Plan initialPlan(List initialTowers, List finalTowers) {
	PlanBuilder builder = new SimplePlanBuilder();
	builder.addActivity(new Activity(pattern("(stacking-problem)")));
	return builder.getPlan();
    }

    /*
     * Domain
     */

    public Domain problemDomain(List initialTowers, List finalTowers) {
	Domain domain = new Domain();
	domain.addRefinement(problemTaskRefinement());
	domain.addRefinement(initialStateRefinement(initialTowers));
	domain.addRefinement(goalStateRefinement(finalTowers));
	return domain;
    }

    Refinement problemTaskRefinement() {
	Refinement r = new Refinement("stacking-problem",
				      pattern("(stacking-problem)"));
	r.setNodes
          (Lisp.list
	    (new NodeSpec("1", pattern("(setup-problem)")),
	     new NodeSpec("2", pattern("(problem-goal)"))));
	r.setOrderings
	  (SimpleDomainEditor.sequentialOrderings(r.getNodes()));
	return r;
    }

    static final Symbol 
	S_ACHIEVE  = Symbol.intern("achieve"),
	S_ON       = Symbol.intern("on"),
	S_BLOCK    = Symbol.intern("block"),
	S_CLEARTOP = Symbol.intern("cleartop"),
	S_TABLE    = Symbol.intern("table"),
	S_TRUE     = Symbol.intern("true"),
	S_FALSE    = Symbol.intern("false");

    Refinement initialStateRefinement(List initialTowers) {
	Refinement r = new Refinement("setup-problem",
				      pattern("(setup-problem)"));
	ListOfConstraint effects = new LinkedListOfConstraint();
	// For each block, B, (block B) = true.
	Set blocks = allBlocks(initialTowers);
	for (Iterator i = blocks.iterator(); i.hasNext();) {
	    Symbol block = (Symbol)i.next();
	    LList blockPat = Lisp.list(S_BLOCK, block);
	    effects.add(worldEffect(blockPat));
	}
	// For each tower, ON and CLEARTOP effects that describe it.
	for (Iterator i = initialTowers.iterator(); i.hasNext();) {
	    List tower = (List)i.next();
	    // Cleartop effect
	    Symbol topBlock = (Symbol)tower.get(0);
	    LList clearPat = Lisp.list(S_CLEARTOP, topBlock);
	    effects.add(worldEffect(clearPat));
	    // ON effects
	    for (Iterator j = towerPatterns(tower).iterator(); j.hasNext();) {
		LList onPat = (LList)j.next();
		effects.add(worldEffect(onPat));
	    }
	}
	r.setConstraints(effects);
	return r;
    }

    Refinement goalStateRefinement(List finalTowers) {
	Refinement r = new Refinement("problem-goal",
				      pattern("(problem-goal)"));
	// For each tower, (ACHIEVE (ON ...)) actions that construct it.
	ListOfNodeSpec nodes = new LinkedListOfNodeSpec();
	int nodeID = 1;
	for (Iterator i = finalTowers.iterator(); i.hasNext();) {
	    List tower = (List)i.next();
	    List onPatterns = towerPatterns(tower);
	    Collections.reverse(onPatterns);
	    for (Iterator j = onPatterns.iterator(); j.hasNext();) {
		LList onPat = (LList)j.next();
		LList achievePat = Lisp.list(S_ACHIEVE, onPat);
		nodes.add(new NodeSpec(nodeID++, achievePat));
	    }
	}
	// Make the nodes sequential.
	r.setNodes(nodes);
	r.setOrderings
	  (SimpleDomainEditor.sequentialOrderings(r.getNodes()));
	return r;
    }

    LList pattern(String pat) {
	return (LList)Lisp.readFromString(pat);
    }

    Set allBlocks(List towerLists) {
	Set result = new TreeSet();
	for (Iterator i = towerLists.iterator(); i.hasNext();)
	    result.addAll((List)i.next());
	return result;
    }

    Constraint worldEffect(LList pattern) {
	return new Constraint("world-state", "effect",
			      Lisp.list(new PatternAssignment(pattern)));
    }

    // Returns a list of PatternAssignments descibing a tower.
    // Includes only the (ON block support) constraints,
    // not the CLEARTOPs.  That's because CLEARTOPs aren't
    // included in the goals.
    List towerPatterns(List tower) {
	List result = new LinkedList();
	Iterator i = tower.iterator();
	Symbol block = (Symbol)i.next();
	while (i.hasNext()) {
	    Symbol support = (Symbol)i.next();
	    result.add(Lisp.list(S_ON, block, support));
	    block = support;
	}
	result.add(Lisp.list(S_ON, block, S_TABLE));
	return result;
    }

    /*
     * World state checking
     */

    public void checkTowerState(Map state, List towers) {
	List incorrectPatterns = incorrectTowerPatterns(state, towers);
	if (incorrectPatterns.isEmpty())
	    Debug.noteln("World state is correct");
	else
	    Debug.noteln("Incorrect patterns", incorrectPatterns);
    }

    public List incorrectTowerPatterns(Map state, List towers) {
	List incorrectPatterns = new LinkedList();
	// For each tower, check ON and CLEARTOP patterns.
	for (Iterator i = towers.iterator(); i.hasNext();) {
	    List tower = (List)i.next();
	    // CLEARTOP should be true for the tops of towers.
	    Symbol topBlock = (Symbol)tower.get(0);
	    LList clearPat = Lisp.list(S_CLEARTOP, topBlock);
	    checkPattern(state, clearPat, S_TRUE, incorrectPatterns);
	    // ON should be true for adjacent towers in blocks.
	    for (Iterator j = towerPatterns(tower).iterator(); j.hasNext();) {
		LList onPat = (LList)j.next();
		checkPattern(state, onPat, S_TRUE, incorrectPatterns);
		// CLEARTOP should be false for blocks that have
		// something on them; however, we don't initially
		// put in all CLEARTOPs that should be false,
		// and so some value == null ones may remain. /\/
		Symbol under = (Symbol)onPat.get(2);
		LList clrPat = Lisp.list(S_CLEARTOP, under);
		if (under != S_TABLE && state.get(clrPat) != null)
		    checkPattern(state, clrPat, S_FALSE, incorrectPatterns);
	    }
	}
	return incorrectPatterns;
    }

    void checkPattern(Map state,
		      LList pattern, Symbol expectedValue,
		      List incorrectPatterns) {
	Object value = state.get(pattern);
	if (value == null || value != expectedValue) {
	    incorrectPatterns.add(pattern);
	}
    }

}
