/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Apr  4 21:03:04 2005 by Jeff Dalton
 * Copyright: (c) 2001 - 2005, AIAI, University of Edinburgh
 */

package ix.iview;

import java.util.*;

import java.awt.Color;
import java.awt.Container;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.Dimension;
import java.awt.event.*;

import javax.swing.*;

import java.io.*;

import ix.icore.IXAgent;
import ix.icore.domain.*;
import ix.iface.domain.*;

import ix.iface.util.ToolFrame;
import ix.iface.util.VerticalPanel;
import ix.iface.util.RadioButtonBox;
import ix.iface.util.IconImage;
import ix.iface.util.CatchingActionListener;

import ix.util.*;
import ix.util.lisp.*;


/**
 * A simple domain editor.
 */
public class SimpleDomainEditor
    implements InternalDomainEditor, ActionListener {

    JFrame frame;
    Container contentPane;

    JMenu refinementMenu;

    RefinementEditor refinementEditor;

    IXAgent agent;
    Domain dom;

    public SimpleDomainEditor(IXAgent agent, Domain dom) {
	this.agent =agent;
	this.dom = dom;
	setUpFrame();
    }

    protected void setUpFrame() {
	frame = new ToolFrame(agent.getAgentDisplayName() + " Domain Editor");
	frame.setIconImage(IconImage.getIconImage(this));
	frame.setSize(500, 300);
	frame.setJMenuBar(makeMenuBar());
	contentPane  = frame.getContentPane();
	{
	    // The following 2 lines can be commented-out to have the
	    // editor show as mostly blank until the user decides
	    // to edit a refinement.
	    ensureRefinementEditor();
	    refinementEditor.editNewRefinement();
	}
	frame.setVisible(true);
    }

    public void setVisible(boolean v) {
	frame.setVisible(v);
    }

    public void setLocation(int x, int y) {
	frame.setLocation(x, y);
    }

    public void saveExpansion(Refinement r) {
	ensureRefinementEditor();
	refinementEditor.editPartialRefinement(r);
    }

    protected void ensureRefinementEditor() {
	if (refinementEditor == null) {
	    refinementEditor = this.new RefinementEditor();
	    refinementEditor.setBorder(
	        BorderFactory.createTitledBorder("Refinement"));
	    // contentPane.add(refinementEditor, BorderLayout.CENTER);
	    // contentPane.add(refinementEditor);
	    frame.setContentPane(refinementEditor);
	    frame.pack();
	}
    }

    /*
     * Menu bar
     */

    protected JMenuBar makeMenuBar() {
	JMenuBar bar = new JMenuBar();

	// File menu
	JMenu fileMenu = new JMenu("File");
	bar.add(fileMenu);
	fileMenu.add(makeMenuItem("Load Domain"));
	fileMenu.add(makeMenuItem("Check Domain"));
	fileMenu.getMenuComponent(fileMenu.getMenuComponentCount()-1)
	    .setEnabled(false);		// can't check yet
	fileMenu.add(makeMenuItem("Clear Domain"));
	fileMenu.add(makeMenuItem("Save Domain As ..."));
	fileMenu.addSeparator();
	fileMenu.add(makeMenuItem("Close"));

	// Edit menu
	JMenu editMenu = new JMenu("Edit");
	bar.add(editMenu);

	refinementMenu = new JMenu("Refinement");
	populateRefinementMenu();
	editMenu.add(refinementMenu);

	editMenu.add(makeMenuItem("New Refinement"));

	// View menu
	JMenu viewMenu = new JMenu("View");
	bar.add(viewMenu);
	viewMenu.add(makeMenuItem("Simple"));
	viewMenu.add(makeMenuItem("Advanced"));
	viewMenu.getMenuComponent(viewMenu.getMenuComponentCount()-1)
	    .setEnabled(false);		// can't go to advanced editor yet

	return bar;
    }

    protected JMenuItem makeMenuItem(String text) {
	JMenuItem item = new JMenuItem(text);
	item.addActionListener(CatchingActionListener.listener(this));
	return item;
    }

    protected void populateRefinementMenu() {
	refinementMenu.setEnabled(false);
	refinementMenu.removeAll();
	List refinements = dom.getRefinements();
	// Put refinements in alphabetical order by name.
	// sort modifies, so 1st make a copy
	List sorted = new ArrayList(refinements);
	Collections.sort(
	    sorted,
	    new Comparator() {
		public int compare(Object a, Object b) {
		    Refinement r1 = (Refinement)a;
		    Refinement r2 = (Refinement)b;
		    return r1.getName().compareTo(r2.getName());
		}
	    });
	if (!sorted.isEmpty()) {
	    int max = 20;
	    int remaining = sorted.size();
	    int count = 0;
	    JMenu menu = refinementMenu;
	    for (Iterator i = sorted.iterator(); i.hasNext();) {
		if (count == max && remaining > 5) {
		    JMenu submenu = new JMenu("More");
		    menu.add(submenu);
		    menu = submenu;
		    count = 0;
		}
		Refinement r = (Refinement)i.next();
		JMenuItem item = makeMenuItem(r.getName());
		item.setActionCommand("Edit Refinement");
		menu.add(item);
		remaining--;
		count++;
	    }
	    refinementMenu.setEnabled(true);
	}
    }

    /*
     * Action interpreter
     */

    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("DomainEditor action:", command);
	if (command.equals("Close")) {
	    frame.setVisible(false);
	}
	else if (command.equals("Load Domain")) {
	    DomainParser.loadDomain(frame, dom);
	    populateRefinementMenu();
	}
	else if (command.equals("Save Domain As ...")) {
	    DomainWriter.saveDomain(frame, dom);
	}
	else if (command.equals("Clear Domain")) {
	    clearDomain();
	}
	else if (command.equals("Edit Refinement")) {
	    JMenuItem item = (JMenuItem)e.getSource();
	    String refinementName = item.getText();
	    Debug.noteln("Edit refinement", refinementName);
	    ensureRefinementEditor();
	    refinementEditor.editRefinement
		(dom.getNamedRefinement(refinementName));
	}
	else if (command.equals("New Refinement")) {
	    ensureRefinementEditor();
	    refinementEditor.editNewRefinement();
	}
	else
	    Debug.noteln("Nothing to do for", command);
    }


    /*
     * Clearing a domain
     */

    protected void clearDomain() {
	if (dom.isEmpty()) {
	    JOptionPane.showMessageDialog(frame,
		"The domain is already empty.",
		"Empty Domain",
		JOptionPane.INFORMATION_MESSAGE);
	    return;
	}
	switch(JOptionPane.showConfirmDialog(frame,
	           "Are you sure you want to clear the domain?",
		   "Confirm",
		   JOptionPane.YES_NO_OPTION)) {
	case JOptionPane.YES_OPTION:
	    // Continue
	    break;
	case JOptionPane.NO_OPTION:
	    // Return now and don't clear.
	    return;
	}
	// Clear the domain.
	dom.clear();
	populateRefinementMenu();
    }


    /**
     * Used to indicate why an editing command cannot be carried out.
     */
    class EditException extends RuntimeException {
	EditException(String message) { super(message); }
    }


    /*
     * Some static utilities that would be in the RefinementEditor class
     * if inner classes could have static declarations.
     */

    public static ListOfNodeSpec parseNodes(String expansion) {
	ListOfNodeSpec nodes = new LinkedListOfNodeSpec();
	long i = 1;
	// For some reason, it doesn't work on Windows to
	// use lineSpearator here.
	while (!expansion.equals("")) {
	    // String[] parts =
	    //    Strings.breakAtFirst(lineSeparator, expansion);
	    String[] parts =
		Strings.breakAtFirst("\n", expansion);
	    Long index = new Long(i);
	    LList childPattern = PatternParser.parse(parts[0]);
	    if (!childPattern.isNull()) {
		nodes.add(new NodeSpec(index, childPattern));
		i++;
	    }
	    expansion = parts[1];
	}
	return nodes;
    }

    public static ListOfOrdering sequentialOrderings(List nodes) {
	ListOfOrdering orderings = new LinkedListOfOrdering();
	if (nodes == null || nodes.size() < 2)
	    return orderings;
	Iterator n = nodes.iterator();
	for (NodeSpec from = (NodeSpec)n.next(),
		      to = (NodeSpec)n.next();
	         true;
	         from = to, to = (NodeSpec)n.next()) {
	    orderings
		.add(new Ordering(new NodeEndRef(End.END, from.getId()),
				  new NodeEndRef(End.BEGIN, to.getId())));
	    if (!n.hasNext())
		return orderings;
	}
	// Compiler notices this is unreachable.
    }

    public static boolean orderingsAreSequential(List nodes,
						 List orderings) {
	// N.B. degenerate cases (less than 2 nodes) count as true.
	// /\/: What about orderings from begin to end of the same node?
	// For now, we can assume they do not exist.
	List sequential = sequentialOrderings(nodes);
	return (orderings == null && sequential.isEmpty())
	    || (orderings.size() == sequential.size()
		&& orderings.containsAll(sequential));
    }


    /**
     * Simple refinement-editing panel.
     */
    protected class RefinementEditor extends VerticalPanel // was JPanel
    implements ActionListener {

	String lineSeparator = System.getProperty("line.separator");

	int textCols = 50;
	JTextField nameText;
	JTextField patternText;
	JTextArea expansionText;
	JTextArea commentText;
        TemporalConstraintPanel constraintPanel;

	JButton saveButton;
	JButton deleteButton;
	JButton clearButton;

	Refinement editingRefinement = null;

	RefinementEditor() {
	    super();
	    // setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
	    setUp();
	}

	protected void setUp() {
	    nameText = new JTextField(textCols);
	    patternText = new JTextField(textCols);
	    expansionText = new JTextArea(7, textCols);
	    commentText = new JTextArea(5, textCols);
	    constraintPanel = new TemporalConstraintPanel();

	    addLeftLabel("Name");
	    addFixedHeight(nameText);

	    addLeftLabel("Pattern");
	    addFixedHeight(patternText);

	    addLeftLabel("Expansion");
	    add(new JScrollPane(expansionText));

	    addLeftLabel("Constraints");
	    addFixedHeight(constraintPanel);

	    addLeftLabel("Annotations");
	    add(new JScrollPane(commentText));

	    saveButton = makeButton("Define Refinement");
	    deleteButton = makeButton("Delete Refinement");
	    clearButton = makeButton("Clear");
	    setButtonsEnabled(false);

	    Box buttons = Box.createHorizontalBox();
	    buttons.add(Box.createHorizontalGlue());
	    buttons.add(saveButton);
	    buttons.add(deleteButton);
	    buttons.add(clearButton);
	    buttons.add(Box.createHorizontalGlue());
	    addFixedHeight(buttons);
	}

	protected void addLeftLabel(String text) {
	    // /\/: Might be better to change the AlignmentX of the other
	    // things in the vertical box to be Component.LEFT_ALIGNMENT.
	    // /\/: The above comment no longer applies, because we're
	    // in a VerticalPanel.
	    Box b = Box.createHorizontalBox();
	    b.add(new JLabel(text));
	    b.add(Box.createHorizontalGlue());
	    addFixedHeight(b);
	}

	protected JButton makeButton(String text) {
	    JButton button = new JButton(text);
	    button.addActionListener(CatchingActionListener.listener(this));
	    return button;
	}

	protected void setButtonsEnabled(boolean value) {
	    saveButton.setEnabled(value);
	    deleteButton.setEnabled(value);
	    clearButton.setEnabled(value);
	}

	public void actionPerformed(ActionEvent e) {
	    String command = e.getActionCommand();
	    Debug.noteln("RefinementEditor action:", command);
	    if (command.equals("Define Refinement")) {
		saveDefinition();
	    }
	    else if (command.equals("Delete Refinement")) {
		deleteRefinement();
	    }
	    else if (command.equals("Clear")) {
		clearEdit();
	    }
	    else
		Debug.noteln("Nothing to do for", command);
	}

	public void editNewRefinement() {
	    editingRefinement = null;

	    nameText.setText("");
	    patternText.setText("");
	    expansionText.setText("");
	    commentText.setText("");
	    constraintPanel.newOrderings();
	    setButtonsEnabled(true);
	    deleteButton.setEnabled(false);
	}

	public void editRefinement(Refinement r) {
	    Debug.noteln("Editing refinement named", r.getName());

	    editingRefinement = r;
	    loadFromRefinement(r);

	    setButtonsEnabled(true);
	}

	public void editPartialRefinement(Refinement r) {
	    // Get here via saveExpansion
	    // N.B. We do not do editingRefinement = s.
	    loadFromRefinement(r);
	    setButtonsEnabled(true);
	    deleteButton.setEnabled(false);
	}

	protected void clearEdit() {
	    editNewRefinement();
	    // setButtonsEnabled(false);
	}

	/**
	 * Define a refinement
	 */

	protected void saveDefinition() {
	    if (editingRefinement == null) {
		defineNewRefinement();
	    }
	    else {
		String oldName = editingRefinement.getName();
		String newName = nameText.getText();
		if (newName.equals(oldName)) {
		    // If the name hasn't changed, the user probably wants
		    // to redefine the refinement.
		    switch(JOptionPane.showConfirmDialog(frame,
		               "Redefine refinement " + Util.quote(oldName),
		               "Confirm",
			       JOptionPane.YES_NO_OPTION)) {
		    case JOptionPane.YES_OPTION:
			// Redefine the refinement.
			redefineRefinement(editingRefinement);
			break;
		    case JOptionPane.NO_OPTION:
			// Do nothing.
		    }
		}
		else {
		    // If the name has changed, the user probably wants
		    // to define a new Refinement.
		    Object[] message = {
			"You have changed the name from " 
			    + Util.quote(oldName) + ".",
			"Do you want to define a new refinement named "
			    + Util.quote(newName) + "?"
		    };
		    switch(JOptionPane.showConfirmDialog(frame,
			      message, "Confirm",
			      JOptionPane.YES_NO_OPTION)) {
		    case JOptionPane.YES_OPTION:
			// Define a new refinement based on old one
			defineRenamedRefinement();
			break;
		    case JOptionPane.NO_OPTION:
			// Do nothing.
		    }
		}
	    }
	}

	protected void defineRenamedRefinement() {
	    // Warn user about data loss.
	    LinkedList lossage = new LinkedList();
	    Refinement r = editingRefinement;
	    // Var dcls are lost, but we recompute them anyway.
 	    // if (!Collect.isEmpty(r.getVariableDeclarations()))
 	    //     lossage.add("  Variable declarations");
	    if (!Collect.isEmpty(r.getOrderings())
		&& !orderingsAreSequential(r.getNodes(), r.getOrderings())) 
		lossage.add("  Nonsequential orderings");
	    if (!Collect.isEmpty(r.getConstraints()))
		lossage.add("  Constraints");
	    if (!Collect.isEmpty(r.getIssues()))
		lossage.add("  Issues");
	    if (!lossage.isEmpty()) {
		lossage.addFirst
		    ("Some of the original refinement will be lost:");
		lossage.add("Do you still want to define the new refinement?");
		switch(JOptionPane.showConfirmDialog(frame,
		          lossage.toArray(), "Confirm",
			  JOptionPane.YES_NO_OPTION)) {
		case JOptionPane.YES_OPTION:
		    // Go through to act like it was a completely
		    // new definition.
		    break;
		case JOptionPane.NO_OPTION:
		    // Do not define.
		    return;
		}
	    }
	    // Define
	    editingRefinement = null;	// forget old refinement
	    saveDefinition(); 		// and try saving again
	}

	protected void defineNewRefinement() {
	    Refinement r;
	    try {
		String name = nameText.getText().trim();
		LList pattern = PatternParser.parse(patternText.getText());
		if (name.equals(""))
		    throw new EditException("No name for refinement.");
		if (pattern.isNull())
		    throw new EditException("No pattern for refinement.");
		if (SimpleDomainEditor.this.dom.getNamedRefinement(name)
		        != null)
		    throw new EditException
			("There is already a refinement named "
			 + Util.quote(name));
		r = new Refinement(name, pattern);
		storeIntoRefinement(r);
	    }
	    catch (Exception e) {
		Debug.noteException(e);
		JOptionPane.showMessageDialog(frame,
		     new Object[] {"Cannot define refinement.",
				   Debug.foldException(e)},
		     "Error while defining refinement",
                     JOptionPane.ERROR_MESSAGE);
		return;
	    }
	    editingRefinement = r;
	    SimpleDomainEditor.this.dom.addRefinement(r);
	    SimpleDomainEditor.this.populateRefinementMenu();
	}

	protected void redefineRefinement(Refinement r) {
	    Debug.noteln("Redefining refinement", r.toString());
	    // Can't yet redefine refinements.  /\/
	    JOptionPane.showMessageDialog(frame,
	         "Refinement redefinition is not yet supported.",
		 "Redefinition not allowed",
		 JOptionPane.ERROR_MESSAGE);
	}

	protected void loadFromRefinement(Refinement r) {
	    // Name might be null because we might get here via saveExpansion.
	    nameText.setText(r.getName() != null ? r.getName() : "");
	    patternText.setText(PatternParser.unparse(r.getPattern()));
	    expansionText.setText("");
	    if (r.getNodes() != null) {
		for (Iterator i = r.getNodes().iterator(); i.hasNext();) {
		    NodeSpec spec = (NodeSpec)i.next();
		    LList childPattern = spec.getPattern();
		    expansionText.append(PatternParser.unparse(childPattern));
		    expansionText.append(lineSeparator);
		}
	    }
	    commentText.setText(r.getComments());
	    constraintPanel.loadFromOrderings(r.getNodes(), r.getOrderings());
	}

	protected void storeIntoRefinement(Refinement r) {
	    // /\/: Eventually, this will have to modify refinements
	    // that already contain some data; but for not it will always
	    // be a new refinement with only the pattern (and values
	    // derived from the pattern) filled-in.

	    Debug.noteln("Changing refinement", r.toString());
	    Debug.expect(r.getName().equals(nameText.getText().trim()));

	    // Nodes
	    String expansion = expansionText.getText().trim();
	    ListOfNodeSpec nodes = parseNodes(expansion);
	    r.setNodes(nodes.isEmpty() ? null : nodes);

	    // Orderings
	    String ordChoice = constraintPanel.getOrderingChoice();
	    ListOfOrdering orderings = ordChoice.equals("Sequential")
		? sequentialOrderings(nodes)
		: null;
	    // Don't change existing orderings if choice is "Other"
	    if (!ordChoice.equals("Other") && orderings != null)
		r.setOrderings(orderings.isEmpty() ? null : orderings);

	    // Comments
	    String comments = commentText.getText();
	    if (!comments.trim().equals(""))
		r.setComments(comments);
	    else
		r.setComments(null);

	    // Variables
	    fixVarDcls(r);

	    // /\/: Probably a bug if this check fails.
	    r.checkConsistency();

	    Debug.noteln("Refinement is now", r.toString());
	}

	protected void fixVarDcls(Refinement r) {
	    Set declared = r.getDeclaredVariables();
	    Set used = r.getVariablesUsed();
	    if (used.equals(declared))
		return;		// ok as-is
	    // We know that if one of used or declared is empty,
	    // the other, since not equal, must not be empty.
	    String declaredNames = Collect.elementsToString(declared);
	    String usedNames = Collect.elementsToString(used);
	    Object message = declared.isEmpty()
		// /\/: For now, only the "add" case occurs, because
		// we don't allow refinement redefinition or retain
		// var dcls when defining a renamed refinement.
		? new String[] {
		    "Need to add declarations for variables: "
		    + usedNames
		}
		: used.isEmpty()
		? new String[] {
		    "Need to remove declarations for variables: "
		    + declaredNames
		}
		: new String[] {
		    "Need to change declared variables",
		    "from " + declaredNames,
		    "to " + usedNames
		};
	    JOptionPane.showMessageDialog(frame,
                message,
	        "Change while defining refinement",
		JOptionPane.INFORMATION_MESSAGE);
	    r.setVariableDeclarations(makeVarDcls(used));
	}

	protected ListOfVariableDeclaration makeVarDcls(Set vars) {
	    ListOfVariableDeclaration result =
		new LinkedListOfVariableDeclaration();
	    for (Iterator i = vars.iterator(); i.hasNext();) {
		ItemVar v = (ItemVar)i.next();
		result.add(new VariableDeclaration(v));
	    }
	    return result.isEmpty() ? null : result;
	}

	/**
	 * Delete a refinement.
	 */
	protected void deleteRefinement() {
	    Debug.expect(editingRefinement != null);
	    String oldName = editingRefinement.getName();
	    String newName = nameText.getText();

	    if (!newName.equals(oldName)) {
		// The user has changed the name
		JOptionPane.showMessageDialog(frame,
		    "You cannot delete the refinement"
		    + " after changing the name.",
		    "Name changed",
		    JOptionPane.ERROR_MESSAGE);
		deleteButton.setEnabled(false);
		return;
	    }

	    if (SimpleDomainEditor.this.dom.getNamedRefinement(newName)
		== null) {
		// There's no refinement with the name that's in the form.
		JOptionPane.showMessageDialog(frame,
		    "There is no refinement named " + Util.quote(newName),
		    "No such refinement",
		    JOptionPane.ERROR_MESSAGE);
		deleteButton.setEnabled(false);
		return;
	    }

	    // Make sure the user really wants to delete it.
	    switch (JOptionPane.showConfirmDialog(frame,
		      "Delete refinement " + Util.quote(newName),
		      "Confirm",
		      JOptionPane.YES_NO_OPTION)) {
	    case JOptionPane.YES_OPTION:
		// Delete the refinement.
		SimpleDomainEditor.this.dom.deleteNamedRefinement(newName);
		SimpleDomainEditor.this.populateRefinementMenu();
		clearEdit();
		break;
	    case JOptionPane.NO_OPTION:
		// Do nothing.
	    }
	}

    }

    /**
     * A subpanel of the refinement editor that handles the choice
     * between parallel and sequential ordering contraints on
     * a refinement's nodes.
     */
    public static class TemporalConstraintPanel extends JPanel {
	RadioButtonBox box = RadioButtonBox.createHorizontalBox();

	JRadioButton ordParallel = new JRadioButton("Parallel", false);
	JRadioButton ordSequential = new JRadioButton("Sequential", true);
	JRadioButton ordOther = new JRadioButton("Other", false);

	public TemporalConstraintPanel() {
	    setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
	    // setBorder(BorderFactory.createLineBorder(Color.gray));
	    // setBackground(Color.white);

	    // Temporal constraints
	    JPanel temporalPanel = new JPanel();
	    temporalPanel
		.setLayout(new BoxLayout(temporalPanel, BoxLayout.X_AXIS));
	    temporalPanel
		.setBorder(BorderFactory.createTitledBorder("Temporal"));

	    ordOther.setEnabled(false);

	    box.add(new JLabel("Activities are", SwingConstants.LEFT));
	    box.add(Box.createHorizontalStrut(10));
	    box.add(ordParallel);
	    box.add(ordSequential);
	    box.add(ordOther);
	    box.add(Box.createHorizontalGlue());

	    temporalPanel.add(box);
	    add(temporalPanel);
	}

	public String getOrderingChoice() {
	    String choice = box.getSelection();
	    Debug.noteln("Ordering choice", choice);
	    Debug.expect(choice != null);
	    return choice;
	}

	public void newOrderings() {
	    ordParallel.setEnabled(true);
	    ordSequential.setEnabled(true);
	    ordOther.setEnabled(false);

	    // ordSequential.setSelected(true);
	    box.setSelection("Sequential");
	}

	public void loadFromOrderings(List nodes, List orderings) {
	    if (orderings == null
		    || orderings.isEmpty()
		    || orderingsAreSequential(nodes, orderings)) {
		// Allow parallel / sequential choice
		ordParallel.setEnabled(true);
		ordSequential.setEnabled(true);
		ordOther.setEnabled(false);
		if ((orderings == null || orderings.isEmpty())
		        && nodes != null && nodes.size() >= 2)
		    // ordParallel.setSelected(true);
		    box.setSelection("Parallel");
		else
		    // ordSequential.setSelected(true);
		    box.setSelection("Sequential");
	    }
	    else {
		// Refinement already has orderings that are not sequential,
		// but still allow parallel / sequential choice.
		ordParallel.setEnabled(true);
		ordSequential.setEnabled(true);
		ordOther.setEnabled(true);
		// ordOther.setSelected(true);
		box.setSelection("Other");
	    }
	}

    }

}

// Issues:
// * Need a way to prevent the JFileChooser from creating new folders.
