/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Apr 25 20:38:24 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2005, AIAI, University of Edinburgh
 */

package ix.itest;

import java.awt.Container;
import java.awt.event.*;

import javax.swing.*;

import java.util.*;

import ix.icore.IXAgent;
import ix.icore.Sendable;
import ix.ichat.ChatMessage;
import ix.icore.event.AgentNameEvent;

import ix.ispace.ISpaceTool;

import ix.iface.util.ToolController;
import ix.iface.util.IconImage;
import ix.iface.util.CatchingActionListener;

import ix.iface.ui.HelpFrame;
// import ix.iface.ui.AboutFrame;  // we don't (yet?) use one /\/

import ix.ip2.PanelFrame;

import ix.util.*;
import ix.util.xml.XML;

/**
 * The main frame of the I-Test user interface.
 */
public class ItestFrame extends PanelFrame implements ActionListener {

    protected Itest itest;
    protected SendPanel sendPanel;
    protected ReceivePanel receivePanel;
    protected JPanel logoPanel;

    /**
     * Creates the frame and adds its menu bar.
     */
    ItestFrame(Itest itest) {
	super(itest);
	this.itest = itest;
    }

    void setup() {
	sendPanel = new ITestSendPanel(itest);
	receivePanel = new ReceivePanel(itest);
	logoPanel = itest.makeLogoPanel();

	setIconImage(IconImage.getIconImage(this));
	setJMenuBar(makeMenuBar());

	Container contentPane = getContentPane();
	contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.Y_AXIS));
	contentPane.add(receivePanel);
	contentPane.add(sendPanel);
	contentPane.add(logoPanel);
    }

    /**
     * Gives the main I-TEST frame its contents
     */
    public void setupFrame() {

    }

    /**
     * Creates the menu bar
     */
    protected JMenuBar makeMenuBar() {
	JMenuBar bar = new JMenuBar();

	JMenu fileMenu = new JMenu("File");
	bar.add(fileMenu);
	fileMenu.add(makeMenuItem("Exit"));

	bar.add(toolManager.getToolsMenu());
	addItestTools();

	JMenu helpMenu = new JMenu("Help");
	bar.add(helpMenu);
	helpMenu.add(makeMenuItem("Help"));
	helpMenu.add(makeMenuItem("About I-Test"));

	bar.add(Box.createHorizontalGlue()); // let later items move right

	bar.add(testMenu);
	testMenu.setEnabled(false); 	// initially

	return bar;
    }

    /**
     * Handles a change in the panel's symbol-name.
     */
    public void symbolNameChanged(AgentNameEvent e) {
	super.symbolNameChanged(e); // sets frame title
	// Change the panel that identifies the system etc.
	Container contentPane = getContentPane();
	contentPane.remove(logoPanel);
	logoPanel = itest.makeLogoPanel();
	contentPane.add(logoPanel);	
	validate();
    }

    /**
     * Action interpreter
     */
    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("I-Test action:", command);
	if (command.equals("Exit")) {
	    if (Util.dialogConfirms(null, "Are you sure you want to exit?"))
		System.exit(0);		// should call agent exit /\/
	}
	else if (command.equals("Help")) {
	    helpController.ensureToolVisible();
	}
	else if (command.equals("About I-Test")) {
	    aboutITestController.ensureToolVisible();
	}
	else
	    Debug.noteln("Nothing to do for", command);
    }

    protected void addItestTools() {
	addTool(new ToolController("I-Space") {
	    public Object createTool() {
		return new ISpaceTool(agent);
	    }
	});
    }

    // Redefine PanelFrame method - used with test-menu entries.
    // See ix.ip2.test.PanelTest.
    public SendPanel ensureSendPanelExists() {
	return sendPanel;
    }

    // This class exists so that we can define a sending method.
    class ITestSendPanel extends SendPanel {
	ITestSendPanel(Itest itest) {
	    super(itest);
	}
	protected void sending(String destination, Sendable contents) {
	    // Record outgoing chat-messages in the transcript.
	    if (contents instanceof ChatMessage)
		receivePanel.newMessage((ChatMessage)contents);
	}
    }

    // /\/: We should simplify the way help and about work and
    // make more in common with I-P2 rather than have duplicate
    // code.

    // /\/: This code is newer than the code in Ip2Frame and
    // is an experiment with using ToolControllers for help and about.

    /*
     * Help
     */

    ToolController helpController = new ToolController("I-Test Help") {
	public Object createTool() {
	    return new HelpFrame("ix-info.html");
	}
	// /\/: It looks like the Ip2 version puts it back on the
	// initial page if the user asks for it again.
    };

    /*
     * About
     */

    ToolController aboutITestController = new ToolController("About I-Test") {
	public Object createTool() {
	    String title = itest.getAgentDisplayName() + " About I-Test";
	    TextAreaFrame about = new ix.ip2.Ip2Frame.Ip2AboutFrame(title);
	    about.setText(makeAboutText());
	    return about;
	}
	public Object ensureToolVisible() {
	    // We have to remake the about text because some
	    // of the information might have changed.
	    TextAreaFrame frame = (TextAreaFrame)ensureTool();
	    frame.setText(makeAboutText());
	    return super.ensureToolVisible();
	}
    };

    String makeAboutText() {
	// /\/: Most of this should be shared with Ip2Frame, perhaps
	// by putting the code in methods of the AboutFrame class.
	List about = new LinkedList();
	// I-X version
	about.add("I-X version: " + ix.Release.version
		  + ", release date " + ix.Release.date);
	// Agent class
	about.add("Agent class: " + itest.getClass().getName());
	// Startup time
	about.add("Started running: " +
		  new ISODateFormat()
		      .formatDateTime(itest.getAgentStartupDate()));
	// Java version
	about.add("Java version: " + System.getProperty("java.version")
		  + " from " + System.getProperty("java.vendor"));
	// Parameters
	Hashtable props = Parameters.getParameters();
	SortedSet keys = new TreeSet(props.keySet());
	about.add("");
	if (keys.isEmpty()) {
	    about.add("Parameters: none");
	}
	else {
	    about.add("Parameters: ");
	    for (Iterator i = keys.iterator(); i.hasNext();) {
		String key = (String)i.next();
		String val = (String)props.get(key);
		about.add("   " + key + " = " + val
			  + (Parameters.usedParameter(key)
			     ? "" : " [unused]"));
	    }
	}
	// Let the controller add things
	// ip2.getController().addAboutInfo(about);
	// Let the file-syntax manager add things
	XML.fileSyntaxManager().addAboutInfo(about);
	// Concat into string
	return Strings.joinLines(about);
    }

}
