/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Sep 27 02:53:02 2002 by Jeff Dalton
 * Copyright: (c) 2002, AIAI, University of Edinburgh
 */

package ix.ispace;

import java.awt.Container;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.*;

import javax.swing.*;

import java.util.*;

import ix.icore.IXAgent;
import ix.icore.Sendable;
import ix.iface.util.CatchingActionListener;

import ix.util.*;

public class GroupSender implements ActionListener {

    protected IXAgent agent;
    protected Sendable contents;

    protected ContactManager contactManager;
    protected List groupNames;

    protected JFrame frame;
    protected Container contentPane;

    protected GroupPanel groupPanel;
    protected ButtonPanel buttonPanel;

    public GroupSender(IXAgent agent, Sendable contents) {
	this.agent = agent;
	this.contents = contents;
	
	contactManager = agent.getContactManager();
	groupNames =
	    (List)Collect.map(contactManager.getAgentGroups(),
			      Fn.accessor(AgentGroup.class, "getName"));

	setUpFrame();
    }

    void setUpFrame() {
	frame = new JFrame(agent.getAgentDisplayName() + " Group Sender");
	contentPane = frame.getContentPane();

	groupPanel = new GroupPanel();
	buttonPanel = new ButtonPanel();

	// default is equiv to setLayout(new BorderLayout());
	contentPane.add(groupPanel, BorderLayout.CENTER);
	contentPane.add(buttonPanel, BorderLayout.SOUTH);

	// frame.pack();
	frame.setSize(300, 200);
	frame.setVisible(true);
    }

    public void actionPerformed(ActionEvent e) {
	String command = e.getActionCommand();
	Debug.noteln("GroupSender action:", command);
	if (command.equals("Send")) {
	    sendToSelectedGroup();
	}
	else if (command.equals("Cancel")) {
	    frame.setVisible(false);
	}
	else
	    Debug.noteln("Nothing to do for", command);
    }

    protected void sendToSelectedGroup() {
	groupPanel.sendingInit();
	buttonPanel.sendingInit();
	new SendingThread().start();
    }

    class SendingThread extends CatchingThread {

	Sendable messageContents;
	Collection toNames;

	int delay = 3 * 1000;	// milliseconds

	SendingThread() {
	    this.toNames = groupPanel.getSelectedNames();
	    this.messageContents = contents;
	}

	public void innerRun() {

	    for (Iterator i = toNames.iterator(); i.hasNext();) {
		final String toName = (String)i.next();

		Debug.noteln("Sending to", toName);
		try {
		    SwingUtilities.invokeAndWait(new Runnable() {
			public void run() {
			    groupPanel.sendingTo(toName);
			}
		    });
		}
		catch (Throwable t) {
		}
		sendTo(toName, messageContents);

		try { Thread.sleep(delay); }
		catch (InterruptedException e) {}
	    }

	    SwingUtilities.invokeLater(new Runnable() {
		public void run() {
		    frame.setVisible(false);
		}
	    });

	}

	void sendTo(final String toName, final Sendable contents) {
	    // Use a separate thread that will deal with any errors
	    // so that the main sending thread can continue to the
	    // next agent.  /\/: Since we wait for that thread to
	    // finish (we call t,join()), all this really accomplishes
	    // is to let the CatchingThread provide the error-reporting
	    // code.
	    Thread t = new CatchingThread() {
		public void innerRun() {
		    IPC.sendObject(toName, messageContents);
		}
	    };
	    t.start();
	    try { t.join(); }
	    catch (InterruptedException e) { }
	}

    }

    class GroupPanel extends JPanel implements ActionListener {

	JComboBox groupChoice = new JComboBox(groupNames.toArray());
	JList memberList = new JList();

	GroupPanel() {
	    // default is flowlayout
	    setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));

	    groupChoice.setBorder(BorderFactory.createTitledBorder("Group"));
	    add(groupChoice);

	    JScrollPane memberScroll = new JScrollPane(memberList);
	    memberScroll
		.setBorder
		    (BorderFactory.createTitledBorder("Members"));
	    add(memberScroll);

	    groupChoice
		.addActionListener
		    (CatchingActionListener.listener(this));
	    groupChoice
		.setSelectedItem(groupNames.get(0));
	}

	public void actionPerformed(ActionEvent e) {
	    Object source = e.getSource();
	    if (source == groupChoice) {
		memberList.setListData(getSelectedNames().toArray());
	    }
	}

	Collection getSelectedNames() {
	    String name = (String)groupChoice.getSelectedItem();
	    Debug.noteln("Selected group", name);
	    return getMemberNames(name);
	}

	Collection getMemberNames(String groupName) {
	    AgentGroup group = contactManager.getAgentGroup(groupName);
	    Collection members = group.getKnownMembers();
	    Debug.noteln("Group members", members);
	    return Collect.map(members,
			       Fn.accessor(AgentData.class, "getName"));
	}

	void sendingInit() {
	    groupChoice.setEnabled(false);
	}

	void sendingTo(String name) {
	    memberList.setSelectedValue(name, true); // shouldScroll == true
	}

    }

    class ButtonPanel extends JPanel {

	JButton sendButton = makeButton("Send");
	JButton cancelButton = makeButton("Cancel");

	ButtonPanel() {
	    // default is flowlayout
	    add(sendButton);
	    add(cancelButton);
	}

	JButton makeButton(String command) {
	    JButton b = new JButton(command);
	    b.addActionListener
		(CatchingActionListener.listener(GroupSender.this));
	    return b;
	}

        void sendingInit() {
	    sendButton.setEnabled(false);
	    cancelButton.setEnabled(false);
	}

    }

}
