package ix.isim.util;

import java.util.*;

/**
 * <p>This class represents a priority queue, that is a queue in which elements
 * can be held and from which they can be retrieved by priority. For this
 * purpose, each object has a priority associated with it that determines when
 * in relation to the other elements in the queue this element should be
 * retrieved. Priorities are <code>long</code>s here. </p>
 *
 * <p>If there are several elements in the queue that have the same priority
 * their order can be managed by adding them to the front or the end of the
 * list of elements at that priority. Similarly, retrival can be from the front
 * or the end of the list of elements with the same priority. </p>
 *
 * @author Gerhard Wickler
 * @version 25.Feb.2005
 **/

public interface LongPriorityQueue {

    ///////////////////////////////
    // public instance functions //
    ///////////////////////////////

    // abstract //

    /**
     * <p>This function adds the given object at the given priority to this
     * priority queue. If there are already elements queued at this priority,
     * the new element will be queued at the front. </p>
     *
     * @param object the <code>Object</code> to be added to this
     * <code>LongPriorityQueue</code>
     * @param priority the priority with which the element is to be queued
     **/
    public void addElementAtFront(Object object, long priority);

    /**
     * <p>This function adds the given object at the given priority to this
     * priority queue. If there are already elements queued at this priority,
     * the new element will be queued at the end. </p>
     *
     * @param object the <code>Object</code> to be added to this
     * <code>LongPriorityQueue</code>
     * @param priority the priority with which the element is to be queued
     **/
    public void addElementAtEnd(Object object, long priority);

    /**
     * <p>This function tests whether this priority queue is empty. </p>
     *
     * @return <code>true</code> if and only if this <code>LongPriorityQueue</code>
     * contains no elements
     **/
    public boolean isEmpty();

    /**
     * <p>This function returns the size of this priority queue. </p>
     *
     * @return the number of elements contained in this
     * <code>LongPriorityQueue</code>
     **/
    public int length();

    /**
     * <p>This function tests whether the given element is currently in this
     * queue at the given priority. </p>
     *
     * @param object the <code>Object</code> to be tested for
     * @param priority the priority at which it should be queued
     * @return whether it is indeed in the queue
     */
    public boolean containsElementAt(Object object, long priority);

    /**
     * <p>This function returns the element in this priority queue that has the
     * lowest priority. If there are several elements queued at that priority,
     * the element at the front of the list is retrieved. An exception is thrown
     * if there is no such element, i.e. if this priority queue is empty. </p>
     *
     * @return the <code>Object</code> that has the lowest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object getLowestFront() throws NoSuchElementException;

    /**
     * <p>This function returns the element in this priority queue that has the
     * lowest priority. If there are several elements queued at that priority,
     * the element at the end of the list is retrieved. An exception is thrown
     * if there is no such element, i.e. if this priority queue is empty. </p>
     *
     * @return the <code>Object</code> that has the lowest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object getLowestEnd() throws NoSuchElementException;

    /**
     * <p>This function returns the element in this priority queue that has the
     * highest priority. If there are several elements queued at that priority,
     * the element at the front of the list is retrieved. An exception is thrown
     * if there is no such element, i.e. if this priority queue is empty. </p>
     *
     * @return the <code>Object</code> that has the highest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object getHighestFront() throws NoSuchElementException;

    /**
     * <p>This function returns the element in this priority queue that has the
     * highest priority. If there are several elements queued at that priority,
     * the element at the end of the list is retrieved. An exception is thrown
     * if there is no such element, i.e. if this priority queue is empty. </p>
     *
     * @return the <code>Object</code> that has the highest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object getHighestEnd() throws NoSuchElementException;

    /**
     * <p>This function removes and returns the element in this priority queue
     * that has the lowest priority. If there are several elements queued at
     * that priority, the element at the front of the list is retrieved. An
     * exception is thrown if there is no such element, i.e. if this priority
     * queue is empty. </p>
     *
     * @return the <code>Object</code> that has the lowest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object removeLowestFront() throws NoSuchElementException;

    /**
     * <p>This function removes and returns the element in this priority queue
     * that has the lowest priority. If there are several elements queued at
     * that priority, the element at the end of the list is retrieved. An
     * exception is thrown if there is no such element, i.e. if this priority
     * queue is empty. </p>
     *
     * @return the <code>Object</code> that has the lowest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object removeLowestEnd() throws NoSuchElementException;

    /**
     * <p>This function removes and returns the element in this priority queue
     * that has the highest priority. If there are several elements queued at
     * that priority, the element at the front of the list is retrieved. An
     * exception is thrown if there is no such element, i.e. if this priority
     * queue is empty. </p>
     *
     * @return the <code>Object</code> that has the highest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object removeHighestFront() throws NoSuchElementException;

    /**
     * <p>This function removes and returns the element in this priority queue
     * that has the highest priority. If there are several elements queued at
     * that priority, the element at the end of the list is retrieved. An
     * exception is thrown if there is no such element, i.e. if this priority
     * queue is empty. </p>
     *
     * @return the <code>Object</code> that has the highest priority in this
     * <code>LongPriorityQueue</code>
     * @exception NoSuchElementException if this <code>LongPriorityQueue</code> is
     * empty
     **/
    public Object removeHighestEnd() throws NoSuchElementException;

    /**
     * <p>This function attempts to remove the given object at the given
     * priority from this priority queue. Whether the removal was successful is
     * returned. </p>
     *
     * @param object the <code>Object</code> to be removed from this
     * <code>LongPriorityQueue</code>
     * @param priority the priority at which the element is currently queued
     * @return <code>true</code> if and only if the object was indeed removed
     * from the queue
     **/
    public boolean removeElementAt(Object object, long priority);

    /**
     * <p>This function returns an iterator of the elements in this priority
     * queue. </p>
     *
     * @return an <code>Iterator</code> for this <code>LongPriorityQueue</code>
     **/
    public Iterator elements();

}
