/* Author: Gerhard Wickler <g.wickler@ed.ac.uk>
 * Updated: Wed Jun 21 17:26:11 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.isim.dynamic;

import java.io.*;
import java.util.*;
import java.lang.Math;

import ix.util.*;

/**
 * <p>Title: I-X Simple Object Animation Tool</p>
 *
 * <p>Copyright: Copyright (c) 2005</p>
 *
 * <p>Company: AIAI, University of Edinburgh</p>
 *
 * @author Gerhard Wickler
 * @version 1.0
 */
public class PointListMoveSpec implements MoveSpec {
    public PointListMoveSpec() {
        try {
            jbInit();
        } catch (Exception ex) {
	    Debug.displayException(ex);
        }
    }

    // earth's mean radius in m
    static final double eRad = 6371000.0d;

    private boolean finished = false;

    private ArrayList lats = new ArrayList();
    private ArrayList longs = new ArrayList();
    private ArrayList accDists = new ArrayList();
    private ArrayList bearings = new ArrayList();
    private double speedInMps = 0.0d;
    private boolean circle = false;

    long startTime = System.currentTimeMillis();

    public double getSpeed() {
        return speedInMps;
    }

    /*public String toString() {
        return "from latitude " + iLat + " longitude " + iLong +
                " bearing " + iBear + " speed " + speedInMps +
                " mps forever";
         }*/

    public void reset() {
        startTime = System.currentTimeMillis();
    }

    public double getCurrentLatitude() {
        double time = ((double) (System.currentTimeMillis() - startTime)) /
                      1000.0d;
        double dist = speedInMps * time;

        // finish at the end or circle:
        double totalDist = ((Double) accDists.get(accDists.size() - 1)).
                           doubleValue();
        if (dist >= totalDist) {
            if (circle) {
                while (dist >= totalDist) {
                    dist -= totalDist;
                }
            } else {
                finished = true;
                return ((Double) lats.get(lats.size() - 1)).doubleValue() /
                        Math.PI * 180.0d;
            }
        }

        // compute which leg we're on:
        int leg;
        for (leg = accDists.size() - 1; leg >= 0; leg--) {
            if (dist >= ((Double) accDists.get(leg - 1)).doubleValue()) {
                break;
            }
        }

        double iLat = ((Double) lats.get(leg - 1)).doubleValue();
        double iLong = ((Double) longs.get(leg - 1)).doubleValue();
        double iBear = ((Double) bearings.get(leg - 1)).doubleValue();
        // d = angular distance covered on earth's surface
        double d = (dist - ((Double) accDists.get(leg - 1)).doubleValue()) /
                   eRad;

        return (Math.asin(Math.sin(iLat) * Math.cos(d) +
                          Math.cos(iLat) * Math.sin(d) * Math.cos(iBear))) /
                Math.PI * 180.0d;
    }

    public double getCurrentLongitude() {
        double time = ((double) (System.currentTimeMillis() - startTime)) /
                      1000.0d;
        double dist = speedInMps * time;

        // finish at the end or circle:
        double totalDist = ((Double) accDists.get(accDists.size() - 1)).
                           doubleValue();
        if (dist >= totalDist) {
            if (circle) {
                while (dist >= totalDist) {
                    dist -= totalDist;
                }
            } else {
                finished = true;
                return ((Double) lats.get(lats.size() - 1)).doubleValue() /
                        Math.PI * 180.0d;
            }
        }

        // compute which leg we're on:
        int leg;
        for (leg = accDists.size() - 1; leg >= 0; leg--) {
            if (dist >= ((Double) accDists.get(leg - 1)).doubleValue()) {
                break;
            }
        }

        double iLat = ((Double) lats.get(leg - 1)).doubleValue();
        double iLong = ((Double) longs.get(leg - 1)).doubleValue();
        double iBear = ((Double) bearings.get(leg - 1)).doubleValue();
        // d = angular distance covered on earth's surface
        double d = (dist - ((Double) accDists.get(leg - 1)).doubleValue()) /
                   eRad;

        return (iLong + Math.atan2(
                Math.sin(iBear) * Math.sin(d) * Math.cos(iLat),
                Math.cos(d) - Math.sin(iLat) * Math.sin(iLat))) /
                Math.PI * 180.0d;
    }

    public boolean finished() {
        return finished;
    }

    private double readNumeric(StreamTokenizer st, String attrName) throws java.
            text.ParseException, java.io.IOException {
        // read the attibute name:
        if ((st.ttype != StreamTokenizer.TT_WORD) ||
            (!st.sval.equals(attrName))) {
            throw new java.text.ParseException(
                    "\"" + attrName + "\" expected", st.lineno());
        }
        st.nextToken();
        if (st.ttype != StreamTokenizer.TT_NUMBER) {
            throw new java.text.ParseException(
                    "numeric value expected", st.lineno());
        }
        double result = st.nval;
        st.nextToken();
        return result;
    }

    static private double getDistance(
            double lat1, double long1, double lat2, double long2) {
        double dLat = lat1 - lat2;
        double dLong = long1 - long2;

        double a = Math.sin(dLat / 2) * Math.sin(dLat / 2) +
                   Math.cos(lat1) * Math.cos(lat2) *
                   Math.sin(dLong / 2) * Math.sin(dLong / 2);
        double c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
        double d = eRad * c;

        return d;
    }

    static private double getBearing(
            double lat1, double long1, double lat2, double long2) {
        double dLat = lat2 - lat1;
        double dLong = long2 - long1;

        double b = Math.atan2(
                Math.sin(dLong) * Math.cos(lat2),
                Math.cos(lat1) * Math.sin(lat2) -
                Math.sin(lat1) * Math.cos(lat2) * Math.cos(dLong));

        return b;
    }

    public PointListMoveSpec(StreamTokenizer st) throws java.text.
            ParseException, java.io.IOException {
        double lastLat = 0.0d;
        double lastLong = 0.0d;
        double currLat = 0.0d;
        double currLong = 0.0d;

        accDists.add(new Double(0.0d));

        // read at least two lat-long pairs:
        for (int i = 1; i <= 2; i++) {
            lastLat = currLat;
            lastLong = currLong;
            // read the latitude:
            currLat = readNumeric(st, "latitude") / 180.0d * Math.PI;
            lats.add(new Double(currLat));
            // read the longitude:
            currLong = readNumeric(st, "longitude") / 180.0d * Math.PI;
            longs.add(new Double(currLong));
        }
        double distance = getDistance(currLat, currLong, lastLat, lastLong);
        accDists.add(new Double(distance));
        bearings.add(new Double(
                getBearing(lastLat, lastLong, currLat, currLong)));
        Debug.noteln("bearing: " +
		     getBearing(lastLat, lastLong, currLat, currLong));

        // read more lat-long pairs if available:
        while ((st.ttype == StreamTokenizer.TT_WORD) &&
               (st.sval.equals("latitude"))) {
            lastLat = currLat;
            lastLong = currLong;
            // read the latitude:
            currLat = readNumeric(st, "latitude") / 180.0d * Math.PI;
            lats.add(new Double(currLat));
            // read the longitude:
            currLong = readNumeric(st, "longitude") / 180.0d * Math.PI;
            longs.add(new Double(currLong));

            distance += getDistance(currLat, currLong, lastLat, lastLong);
            accDists.add(new Double(distance));
            bearings.add(new Double(
                    getBearing(lastLat, lastLong, currLat, currLong)));
        }

        // read the speed:
        double speedValue = readNumeric(st, "speed");
        // read the speed unit:
        if (st.ttype != StreamTokenizer.TT_WORD) {
            throw new java.text.ParseException(
                    "unit for speed expected", st.lineno());
        }
        if (st.sval.equals("mps")) {
            speedInMps = speedValue;
        } else if (st.sval.equals("kmph")) {
            speedInMps = speedValue / 3.6;
        } else {
            throw new java.text.ParseException(
                    "unknown unit for spped", st.lineno());

        }
        st.nextToken();
        // read the stop token:
        if ((st.ttype != StreamTokenizer.TT_WORD) ||
            ((!st.sval.equals("stop")) && (!st.sval.equals("circle")))) {
            throw new java.text.ParseException(
                    "\"stop\" or \"circle\" expected", st.lineno());
        }
        if (st.sval.equals("circle")) {
            circle = true;
            lastLat = currLat;
            lastLong = currLong;
            // go to the origin:
            currLat = ((Double) lats.get(0)).doubleValue();
            lats.add(lats.get(0));
            currLong = ((Double) longs.get(0)).doubleValue();
            longs.add(longs.get(0));
            // compute the accumulated distance and bearing:
            distance += getDistance(currLat, currLong, lastLat, lastLong);
            accDists.add(new Double(distance));
            bearings.add(new Double(
                    getBearing(lastLat, lastLong, currLat, currLong)));
        }
        st.nextToken();
    }

    private void jbInit() throws Exception {
    }
}
