/* Author: Gerhard Wickler <g.wickler@ed.ac.uk>
 * Updated: Wed Jun 21 17:27:07 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.isim.dynamic;

import java.io.*;
import java.lang.Math;

/**
 * <p>Title: I-X Simple Object Animation Tool</p>
 *
 * <p>Copyright: Copyright (c) 2005</p>
 *
 * <p>Company: AIAI, University of Edinburgh</p>
 *
 * @author Gerhard Wickler
 * @version 1.0
 */
public class BearingMoveSpec implements MoveSpec {

    // earth's mean radius in m
    static final double eRad = 6371000.0d;

    double iLat = 0.0d;
    double iLong = 0.0d;
    double iBear = 0.0d;
    double speedInMps = 0.0d;

    long startTime = System.currentTimeMillis();

    public double getSpeed() {
        return speedInMps;
    }

    public String toString() {
        return "from latitude " + iLat + " longitude " + iLong +
                " bearing " + iBear + " speed " + speedInMps +
                " mps forever";
    }

    public void reset() {
        startTime = System.currentTimeMillis();
    }

    public double getCurrentLatitude() {
        double time = ((double) (System.currentTimeMillis() - startTime)) /
                      1000.0d;
        // d = angular distance covered on earth's surface
        double d = (speedInMps * time) / eRad;

        return (Math.asin(Math.sin(iLat) * Math.cos(d) +
                          Math.cos(iLat) * Math.sin(d) * Math.cos(iBear))) /
                Math.PI * 180.0d;
    }

    public double getCurrentLongitude() {
        double time = ((double) (System.currentTimeMillis() - startTime)) /
                      1000.0d;
        // d = angular distance covered on earth's surface
        double d = (speedInMps * time) / eRad;

        return (iLong + Math.atan2(
                Math.sin(iBear) * Math.sin(d) * Math.cos(iLat),
                Math.cos(d) - Math.sin(iLat) * Math.sin(iLat))) /
                Math.PI * 180.0d;
    }

    public boolean finished() {
        return false;
    }

    private double readNumeric(StreamTokenizer st, String attrName) throws java.
            text.ParseException, java.io.IOException {
        // read the attibute name:
        if ((st.ttype != StreamTokenizer.TT_WORD) ||
            (!st.sval.equals(attrName))) {
            throw new java.text.ParseException(
                    "\"" + attrName + "\" expected", st.lineno());
        }
        st.nextToken();
        if (st.ttype != StreamTokenizer.TT_NUMBER) {
            throw new java.text.ParseException(
                    "numeric value expected", st.lineno());
        }
        double result = st.nval;
        st.nextToken();
        return result;
    }

    public BearingMoveSpec(StreamTokenizer st) throws java.text.
            ParseException, java.io.IOException {
        // read the latitude:
        iLat = readNumeric(st, "latitude") / 180.0d * Math.PI;
        // read the longitude:
        iLong = readNumeric(st, "longitude") / 180.0d * Math.PI;
        // read the direction:
        iBear = readNumeric(st, "bearing") / 180.0d * Math.PI;
        // read the speed:
        double speedValue = readNumeric(st, "speed");
        // read the speed unit:
        if (st.ttype != StreamTokenizer.TT_WORD) {
            throw new java.text.ParseException(
                    "unit for speed expected", st.lineno());
        }
        if (st.sval.equals("mps")) {
            speedInMps = speedValue;
        } else if (st.sval.equals("kmph")) {
            speedInMps = speedValue / 3.6;
        } else {
            throw new java.text.ParseException(
                    "unknown unit for spped", st.lineno());

        }
        st.nextToken();
        // read the forever token:
        if ((st.ttype != StreamTokenizer.TT_WORD) ||
            (!st.sval.equals("forever"))) {
            throw new java.text.ParseException(
                    "\"forever\" expected", st.lineno());
        }
        st.nextToken();
    }
}
