/* Author: Gerhard Wickler <g.wickler@ed.ac.uk>
 * Updated: Wed Jun 21 17:27:26 2006 by Jeff Dalton
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.isim.dynamic;

import java.io.*;
import java.util.*;

import ix.util.*;

/**
 * <p>Title: I-X Simple Object Animation Tool</p>
 *
 * <p>Copyright: Copyright (c) 2005</p>
 *
 * <p>Company: AIAI, University of Edinburgh</p>
 *
 * @author Gerhard Wickler
 * @version 1.0
 */
public class AnimatedObject extends Thread {

    // display resolution in dpi:
    final static int displayRes = 72;
    // dot size on display in metres:
    final static double screenDotSize = 1.0d / (40.0d * displayRes);
    // initially assumed scale:
    final static int iniScale = 2000000;

    private String symName;
    private List receivers = new LinkedList();
    private MoveSpec animation;

    private boolean terminationFlag = false;

    // time between position updates in ms:
    private int msgDelay;

    public void useScale(int scale) {
        // dot size in reality in metres:
        double realDotSize = screenDotSize * scale;
        this.msgDelay = (int) (1000.0d * (realDotSize / animation.getSpeed()));
    }

    public void run() {
        Debug.noteln("starting animation for " + symName);
        animation.reset();
        while ((! animation.finished()) && (! getTerminationFlag())) {
            double currentLat = animation.getCurrentLatitude();
            double currentLong = animation.getCurrentLongitude();
            for (Iterator ri = receivers.iterator(); ri.hasNext(); ) {
                String target = (String) ri.next();
                ObjectAnimator.sendConstraint(
                        "latitude", symName, currentLat, target);
                ObjectAnimator.sendConstraint(
                        "longitude", symName, currentLong, target);
            }
            try {
                this.sleep(msgDelay);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }
        Debug.noteln("animation for " + symName + " finished.");
    }

    public void terminate() {
        Debug.noteln("terminating animation for " + symName);
        setTerminationFlag();
    }

    synchronized private void setTerminationFlag() {
        terminationFlag = true;
    }

    synchronized private boolean getTerminationFlag() {
        return terminationFlag;
   }

   public String toString() {
        String result = "animate " + symName;
        for (ListIterator li = receivers.listIterator(); li.hasNext(); ) {
            result += " receivedBy " + li.next();
        }
        return result + ' ' + animation.toString();
    }

    private void readReceivers(StreamTokenizer st) throws java.text.
            ParseException, java.io.IOException {
        while ((st.ttype == StreamTokenizer.TT_WORD) &&
               (st.sval.equals("receivedBy"))) {
            st.nextToken();
            if (st.ttype != StreamTokenizer.TT_WORD) {
                throw new java.text.ParseException(
                        "receiver name expected", st.lineno());
            }
            receivers.add(st.sval);
            st.nextToken();
        }
    }

    public AnimatedObject(StreamTokenizer st) throws java.text.ParseException,
            java.io.IOException {
        // read the name of the object to be animated:
        if (st.ttype != StreamTokenizer.TT_WORD) {
            throw new java.text.ParseException(
                    "symbolic object name expected", st.lineno());
        }
        symName = st.sval;
        st.nextToken();
        // read the names of the receivers:
        readReceivers(st);
        // read the animation type:
        if (st.ttype != StreamTokenizer.TT_WORD) {
            throw new java.text.ParseException(
                    "movement spec expected", st.lineno());
        }
        if (st.sval.equals("from")) {
            st.nextToken();
            this.animation = new BearingMoveSpec(st);
        } else if (st.sval.equals("routed-via")) {
            st.nextToken();
            this.animation = new PointListMoveSpec(st);
        } else {
            throw new java.text.ParseException(
                    "movement spec expected", st.lineno());
        }
        useScale(iniScale);
    }
}
