/* Author: Gerhard Wickler <g.wickler@ed.ac.uk>
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.isim;

import java.text.ParseException;
import java.util.Iterator;

import ix.ip2.*;
import ix.icore.Activity;
import ix.util.Duration;
import ix.util.lisp.LList;
import ix.util.lisp.Symbol;

public class TimedEvent {

    /** the stutus values for events **/
    final static public int WAITING = 0;
    final static public int EXECUTING = 1;
    final static public int COMPLETED = 2;
    final static public int FAILED = 3;
    final static public String[] statusString =
            new String[] { "WAITING", "EXECUTING", "COMPLETED", "FAILED" };

    /** the "predicate" used for the relative event time **/
    final static private Symbol atTimeSy = Symbol.intern("at-time");

    /** the time after the start of the simulation when this event should
     * occur (given as a duration) **/
    protected long timeInMillis;
    /** the agenda entry for the activity of this event **/
    protected Activity event;
    /** the thread to which this event belongs **/
    protected String thread = null;
    /** the status of this event **/
    protected int status = WAITING;

    private TimedEvent(long time, Activity event) throws SimulationException {
        this.timeInMillis = time;
        this.event = event;
        // make sure the action is ground:
        if (!event.isGround()) {
            throw new SimulationException("Timed event action must be ground.");
        }
    }

    public long getRelativeTime() {
        return timeInMillis;
    }

    protected void setRelativeTime(long delay) {
        timeInMillis = delay;
    }

    public Activity getActivity() {
        return event;
    }

    protected void setThread(String thread) {
        this.thread = thread;
    }

    public String getThread() {
        return thread;
    }

    protected void setStatus(int status) {
        this.status = status;
    }

    public int getStatus() {
        return status;
    }

    public String toString() {
        return "(" + atTimeSy + " " +
                (new Duration(timeInMillis)).toISOString() + " " +
                event.toString() + ")";
    }

    /**
     * <p>This factory method must be used to create new timed events. It
     * attempts to parse the the given LList into a timed event assuming the
     * following syntax: </p>
     *
     * <pre>
     * TimedEvent ::= ( atTimeSy ISO-duration Activity )
     * </pre>
     *
     * <p>A ParseException will be thrown if any of the components do not
     * correspond to the expected syntax. </p>
     *
     * @param list the LList representing the timed event
     * @return TimedEvent the new timed event
     * @throws ParseException if the syntax in incorrect
     */
    static public TimedEvent getTimedEvent(LList list) throws ParseException {
        Iterator lobji = list.iterator();
        // expect to parse the atTimeSy off the list:
        if (!lobji.next().equals(atTimeSy)) {
            throw new ParseException("Timed event spec must start with \"" +
                                     atTimeSy + "\"!", 0);
        }
        // the duration since the start of the simulation (in simulated time):
        Duration time = new Duration(lobji.next().toString());
        // the Activity to be executed:
        Activity event = new Activity((LList) lobji.next());
        // there should be no more elements in the given list:
        if (lobji.hasNext()) {
            throw new ParseException("Too many elements in TimedEvent.", 0);
        }
        try {
            return new TimedEvent(time.asMilliseconds(), event);
        } catch (SimulationException se) {
            throw new ParseException(se.getMessage(), 0);
        }
    }

}
