/* Author: Stephen Potter
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.isim;

import ix.icore.Activity;
import ix.icore.IXAgent;
import ix.util.*;
import ix.util.lisp.*;

public class ISimTimerClient extends ISimTimer {

    // ISIM_AGENT_NAME is some address that we can send messages to.
    private String ISIM_AGENT_NAME;

    protected ISimTimerClient(IXAgent agent) {
        super(agent);
        if (!Parameters.haveParameter("isim-agent-name")) {
            Debug.noteln("Cannot create ISimTimerClient. " +
                         "Parameter \"isim-agent-name\" undefined.");
        } else {
            ISIM_AGENT_NAME = Parameters.getParameter("isim-agent-name");
            register();
        }
    }

    // want to send register method:
    private void register() {
        Activity registerActivity =
                new Activity(Lisp.list(
                        SYNCHRONIZE_LABEL,
                        REGISTER,
                        owner.getAgentSymbolName()));
        IPC.sendObject(ISIM_AGENT_NAME, registerActivity);
    }

    public void handleSynchronizeActivity(Activity synchActivity) {
        Symbol verb = (Symbol) synchActivity.getVerb();
        Debug.expectSame(SYNCHRONIZE_LABEL, verb);

        LList parameters = synchActivity.getParameters();
        String param1 = (String) parameters.get(0);

        if (param1.equalsIgnoreCase(START)) {
            handleStartActivity(parameters);
        } else if (param1.equalsIgnoreCase(PAUSE)) {
            handlePauseActivity(parameters);
        } else if (param1.equalsIgnoreCase(RESUME)) {
            handleResumeActivity(parameters);
        } else if (param1.equalsIgnoreCase(ACCELERATE)) {
            handleAccelerateActivity(parameters);
        } else if (param1.equalsIgnoreCase(STOP)) {
            handleStopActivity(parameters);
        } else {
            Debug.noteln("Unrecognised synchronization activity: " + param1 +
                         ". Ignoring.");
        }
    }

    private void handleStartActivity(LList activityParameters) {
        // expects message of form: start <start-time> <accel-factor>
        String param2 = (String) activityParameters.get(1);
        String param3 = (String) activityParameters.get(2);
        try {
            super.start(Long.parseLong(param2), Double.parseDouble(param3));
        } catch (ISimTimerException e) {
            Debug.noteln(e.getMessage());
        }
    }

    private void handlePauseActivity(LList activityParameters) {
        // expects message of form: pause <sim-time>
        String param2 = (String) activityParameters.get(1);
        setCurrentSimTime(Long.parseLong(param2));
        super.pause();
    }

    private void handleResumeActivity(LList activityParameters) {
        // expects message of form: resume <sim-time>
        String param2 = (String) activityParameters.get(1);
        setCurrentSimTime(Long.parseLong(param2));
        super.resume();
    }

    private void handleAccelerateActivity(LList activityParameters) {
        // expects message of form: accelerate <accel-factor>
        String param2 = (String) activityParameters.get(1);
        super.changeAccelerationFactor(Double.parseDouble(param2));
    }

    private void handleStopActivity(LList activityParameters) {
        // expects no parameters
        super.terminateTimer();
    }
}
