/* Author: Gerhard Wickler <g.wickler@ed.ac.uk>
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.isim;

import java.io.*;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import ix.icore.domain.Refinement;
import ix.iface.util.ToolController;
import ix.isim.util.LongPriorityQueue;
import ix.isim.util.TreeOfListsLongPriorityQueue;
import ix.util.Debug;
import ix.util.lisp.*;

public class ISimEngine extends Thread {

    /** the annotation "predicate" that tells the simulation which class to
     * invoke to simulate an action **/
    final static Symbol executeWithSy = Symbol.intern("execute-with");

    /** the ISim to which this ISimEngine belongs **/
    protected ISim theISim;
    /** the Frame for displaying the events etc. **/
    protected ISimEventsFrame simEventsFrame;

    /** whether the simulation thread is waiting for new events to simulate **/
    private boolean waitingForEvents = true;
    /** whether the simulation has been paused **/
    private boolean simPaused = false;
    /** the initial simulation time (set by the user when the simulation is
     * started); used to shift events that are loaded after the simulation was
     * started **/
    private long simStartTime = 0L;
    /** the queue of timed events; the priority is the relative event time **/
    private LongPriorityQueue waitingEvents =
            new TreeOfListsLongPriorityQueue();
    /** the list of timed events that are currently being executed **/
    private List activeEvents = new ArrayList();
    /** the list of timed events that were executed (completed) **/
    private List completedEvents = new ArrayList();

    /**
     * public constructor just memorizes theISim agent this engine belongs to
     * @param isim the ISim that owns this simulation engine
     */
    public ISimEngine(ISim isim) {
        this.theISim = isim;
        simEventsFrame = new ISimEventsFrame(this);
        final ToolController tc = new ToolController("I-Sim Events") {
            public Object createTool() {
                return simEventsFrame;
            }
        };
        theISim.addTool(tc);
    }

    /**
     * This function returns the the simulated time at which the simulation was
     * started.
     * @return long initial simulated time
     */
    public long getSimStartTime() {
        return simStartTime;
    }

    /**
     * This function loads the timed events from the given Files. Times are
     * interpreted as relative to the start time of the simulation if it hasn't
     * started yet; otherwise they will be relative to the current sim-time.
     * @param events the Files containing the timed events
     */
    public void loadTimedEvents(File[] eventFiles) {
        Thread loader = new TimedEventsLoader(eventFiles);
        loader.start();
        Thread.currentThread().yield();
    }

    private class TimedEventsLoader extends Thread {
        File[] eventFiles;

        public TimedEventsLoader(File[] eventFiles) {
            this.eventFiles = eventFiles;
        }

        public void run() {
            ISimTimer timer = theISim.getISimTimer();
            for (int fileI = 0; fileI < eventFiles.length; fileI++) {
                try {
                    String subject = eventFiles[fileI].getName();
                    subject = subject.substring(0, subject.length() - 4);
                    LispFileReader lReader =
                            new LispFileReader(eventFiles[fileI]);
                    readEventFile(lReader, subject, timer);
                    lReader.close();
                } catch (FileNotFoundException fnfe) {
                    fnfe.printStackTrace();
                } catch (ParseException pe) {
                    pe.printStackTrace();
                } catch (SimulationException se) {
                    se.printStackTrace();
                } catch (IOException ioe) {
                    ioe.printStackTrace();
                }
                this.yield();
            }
            // update the table model:
            updateUI();
        }

        private void readEventFile(
                LispFileReader lReader, String subject, ISimTimer timer) throws
                ParseException, SimulationException {
            Object parsed;
            while ((parsed = lReader.readObject()) != Lisp.EOF) {
                // parse the next LList from the given File into a TimedEvent:
                LList event = theISim.getModelManager().
                              putVariablesInPattern((LList) parsed);
                TimedEvent tEvent = TimedEvent.getTimedEvent(event);
                tEvent.setThread(subject);
                // possibly shift the event to relative to current sim-time:
                if (timer.simulationStarted()) {
                    tEvent.timeInMillis +=
                            (timer.getSimTimeInMillis() - simStartTime) + 1L;
                }
                // add the new timed event to the queue:
                this.yield();
                queueTimedEvent(tEvent);
            }
        }

        private void queueTimedEvent(TimedEvent event) {
            // check event is not in the past in simulated time:
            ISimTimer timer = theISim.getISimTimer();
            if (timer.simulationStarted() &&
                ((timer.getSimTimeInMillis() - simStartTime) -
                 event.timeInMillis > 1000)) {
                Debug.noteln("Adding event in the past: " + event);
            }
            waitingEvents.addElementAtEnd(event, event.timeInMillis);
            // may need to restart the simulation:
            if (!simPaused) {
                reactivateEventQueue(event);
            }
            this.yield();
        }
    }


    synchronized private void reactivateEventQueue(TimedEvent newEvent) {
        if (waitingForEvents) {
            waitingForEvents = false;
            this.notify();
        } else if (waitingEvents.getLowestFront() == newEvent) {
            this.interrupt();
        }
        Thread.currentThread().yield();
    }


    public void startSimulation(long simTime, double factor) {
        simStartTime = simTime;
        ISimTimer timer = theISim.getISimTimer();
        try {
            timer.start(simTime, factor);
            this.start();
            Thread.currentThread().yield();
            this.updateUI();
        } catch (ISimTimerException iste) {
            iste.printStackTrace();
        }
    }

    synchronized protected void updateUI() {
        simEventsFrame.makeRowsFromQueue(
                completedEvents, activeEvents, waitingEvents);
    }

    synchronized protected void updateStatus(TimedEvent event, int status) {
        Debug.noteln(TimedEvent.statusString[status] + ": " + event);
        switch (status) {
        case TimedEvent.EXECUTING:
            activeEvents.add(event);
            break;
        case TimedEvent.FAILED:
        case TimedEvent.COMPLETED:
            activeEvents.remove(event);
            completedEvents.add(event);
            break;
        default:
            throw new IllegalStateException();
        }
        event.setStatus(status);
        simEventsFrame.eventsTableModel.eventUpdated(event);
        Thread.currentThread().yield();
    }

    synchronized public void pauseSimulation() {
        simPaused = true;
        this.interrupt();
    }

    synchronized public void resumeSimulation() {
        simPaused = false;
        this.interrupt();
    }

    synchronized public void clearAllWaitingEvents() {
        waitingEvents = new TreeOfListsLongPriorityQueue();
        this.interrupt();
        updateUI();
    }

    synchronized public void executeNextEvent() {
        TimedEvent nextEvent = (TimedEvent) waitingEvents.getLowestFront();
        nextEvent.setRelativeTime(
                theISim.getISimTimer().getSimTimeInMillis() - simStartTime);
        this.interrupt();
    }

    public void run() {
        ISimTimer timer = theISim.getISimTimer();

        while (true) {
            // wait if the simulation is paused or has no events to simulate:
            if (simPaused || waitingForEvents) {
                try {
                    synchronized (this) {
                        this.wait();
                    }
                } catch (InterruptedException ie) {
                    // queue or timer change
                }
            }
            if (waitingEvents.isEmpty()) {
                Debug.noteln("No events to simulate. Waiting ...");
                waitingForEvents = true;
                continue;
            }
            this.yield();
            if (simPaused) {
                continue;
            }

            // work out how long we have to sleep:
            TimedEvent nextEvent = (TimedEvent) waitingEvents.getLowestFront();
            long waitTime =
                    (long) ((simStartTime + nextEvent.timeInMillis -
                             timer.getSimTimeInMillis()) /
                            timer.getAccelerationFactor());
            if (waitTime > 0L) {
                try {
                    this.sleep(waitTime);
                } catch (InterruptedException ie) {
                    // queue or timer change
                    continue;
                }
            }
            this.yield();

            // take next event and execute it:
            synchronized (this) {
                nextEvent = (TimedEvent) waitingEvents.removeLowestFront();
                updateStatus(nextEvent, TimedEvent.EXECUTING);
                try {
                    // find the ExecutableAction for this event:
                    Refinement method = theISim.getDomain().getNamedRefinement(
                            nextEvent.event.getVerb().toString());
                    if (method == null) {
                        throw new SimulationException(
                                "No executable refinement with name " +
                                nextEvent.event.getVerb());
                    }
                    Symbol execClass =
                            (Symbol) method.getAnnotation(executeWithSy);
                    if (execClass == null) {
                        throw new SimulationException(
                                "No ExecutableAction class specified for " +
                                method.getName());
                    }
                    ExecutableAction action =
                            (ExecutableAction) Class.forName(execClass.toString()).
                            newInstance();
                    action.simulate(nextEvent, method);
                    this.yield();

                } catch (SimulationException se) {
                    se.printStackTrace();
                    updateStatus(nextEvent, TimedEvent.FAILED);
                } catch (InstantiationException ie) {
                    ie.printStackTrace();
                    updateStatus(nextEvent, TimedEvent.FAILED);
                } catch (IllegalAccessException iae) {
                    iae.printStackTrace();
                    updateStatus(nextEvent, TimedEvent.FAILED);
                } catch (ClassNotFoundException cnfe) {
                    cnfe.printStackTrace();
                    updateStatus(nextEvent, TimedEvent.FAILED);
                }
                this.yield();
            }
        }

    }
}
