/* Author: Gerhard Wickler <g.wickler@ed.ac.uk>
 * Copyright: (c) 2006, AIAI, University of Edinburgh
 */

package ix.isim;

import java.util.List;

import ix.icore.IXAgent;
import ix.icore.domain.Refinement;
import ix.ip2.Ip2ModelManager;
import ix.util.match.MatchEnv;

public abstract class ExecutableAction {

    protected TimedEvent action;
    protected Refinement method;

    private MatchEnv env = null;

    abstract public void executeAction(MatchEnv env) throws SimulationException;

    protected void simulate(TimedEvent action, Refinement method) {
        this.action = action;
        this.method = method;
        // create and start a thread in which to execute:
        ActionExecutionThread exec = new ActionExecutionThread();
        exec.start();
        // give the new thread time to start:
        Thread.currentThread().yield();
    }

    private MatchEnv verifyConditions() throws SimulationException {
        // check that the preconditions are satisfied:
        Ip2ModelManager mm = ((ISim) IXAgent.getAgent()).getIp2ModelManager();
        List envs = mm.satisfyRefinementPreconditions(action.event, method);
        if ((envs == null) || (envs.size() != 1)) {
            throw new SimulationException(
                    "Preconditions not uniquely satisfiable.");
        }
        MatchEnv env = (MatchEnv) envs.get(0);
        method = mm.fillInRefinement(method, env);
        return env;
    }

    private void finishAction(MatchEnv env) throws SimulationException {
        ISim isim = (ISim) IXAgent.getAgent();
        // assert the effects in the state:
        Ip2ModelManager mm = isim.getIp2ModelManager();
        mm.executeRefinementEffects(method, env);
        // report that the action is finished:
        isim.simulator.updateStatus(action, TimedEvent.COMPLETED);
    }

    public boolean equals(ExecutableAction other) {
        return action.equals(other.action) &&
                method.equals(other.method) &&
                (((env == null) && (other.env == null)) ||
                 env.equals(other.env));
    }

    class ActionExecutionThread extends Thread {
        public void run() {
            try {
                env = verifyConditions();
                this.yield();
                executeAction(env);
                this.yield();
                finishAction(env);
            } catch (SimulationException se) {
                se.printStackTrace();
                ((ISim) IXAgent.getAgent()).simulator.
                        updateStatus(action, TimedEvent.FAILED);
            }
        }
    }
}
