/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu May 12 21:09:49 2005 by Jeff Dalton
 * Copyright: (c) 2005, AIAI, University of Edinburgh
 */

package ix.iscript;

import java.net.URL;
import java.io.*;
import java.util.*;

import ix.util.*;
import ix.util.xml.XML;
import ix.util.lisp.*;

/**
 * A parser for the I-Script language.  It converts an IScript
 * {@link Expression} to an instance of an appropriate subclass
 * of {@link Interpreter.Expr}.  It can also begin with a textual
 * XML representation of an {@link Expression}.
 */
public class IScriptParser {

    public IScriptParser() {
    }

    public Interpreter.Expr parseExpression(String xml) {
	Object obj = XML.objectFromXML(xml);
	Expression expr =(Expression)Util.mustBe(Expression.class, obj);
	return parseExpression(expr);
    }

    public Interpreter.Expr parseExpression(Expression expr) {
        Interpreter.Expr result = do_parseExpression(expr);
	result.setDescription(expr);
	return result;
    }

    Interpreter.Expr do_parseExpression(Expression expr) {
	if (expr instanceof Literal)
	    return parseLiteral((Literal)expr);
	else if (expr instanceof VarRef)
	    return parseVarRef((VarRef)expr);
	else if (expr instanceof Call)
	    return parseCall((Call)expr);
	else if (expr instanceof Assignment)
	    return parseAssignment((Assignment)expr);
	else if (expr instanceof If)
	    return parseIf((If)expr);
	else if (expr instanceof Sequence)
	    return parseSequence((Sequence)expr);
	else if (expr instanceof And)
	    return parseAnd((And)expr);
	else if (expr instanceof Or)
	    return parseOr((Or)expr);
	else if (expr instanceof Let)
	    return parseLet((Let)expr);
	else if (expr instanceof Lambda)
	    return parseLambda((Lambda)expr);
	else if (expr instanceof While)
	    return parseWhile((While)expr);
	else
	    throw new ConsistencyException
		("Unknown I-Script expression type", expr);
    }

    List parseList(ListOfExpression exprs) {
	List result = new LinkedList();
	for (Iterator i = exprs.iterator(); i.hasNext();) {
	    result.add(parseExpression((Expression)i.next()));
	}
	return result;
    }

    Interpreter.Literal parseLiteral(Literal lit) {
	return new Interpreter.Literal(lit.getValue());
    }

    Interpreter.VarRef parseVarRef(VarRef ref) {
	return new Interpreter.VarRef(Symbol.intern(ref.getName().toString()));
    }

    Interpreter.Call parseCall(Call expr) {
	return new Interpreter.Call
	    (parseExpression(expr.getFunction()),
	     parseList(expr.getArguments()));
    }

    Interpreter.Assignment parseAssignment(Assignment expr) {
	return new Interpreter.Assignment
	    (Symbol.intern(expr.getTo().toString()),
	     parseExpression(expr.getValue()));
    }

    Interpreter.If parseIf(If expr) {
	return new Interpreter.If
	    (parseExpression(expr.getTest()),
	     parseExpression(expr.getIfTrue()),
	     parseExpression(expr.getIfFalse()));
    }

    Interpreter.Sequence parseSequence(Sequence expr) {
	return new Interpreter.Sequence(parseList(expr.getOf()));
    }

    Interpreter.And parseAnd(And expr) {
	return new Interpreter.And(parseList(expr.getOf()));
    }

    Interpreter.Or parseOr(Or expr) {
	return new Interpreter.Or(parseList(expr.getOf()));
    }

    Interpreter.Let parseLet(Let expr) {
	List varList = new LinkedList();
	List valExprList = new LinkedList();
	for (Iterator i = expr.getBindings().iterator(); i.hasNext();) {
	    Binding b = (Binding)i.next();
	    varList.add(Symbol.intern(b.getName().toString()));
	    valExprList.add(parseExpression(b.getValue()));
	}
	return new Interpreter.Let
	    (varList,
	     valExprList,
	     parseExpression(expr.getIn()));
    }

    Interpreter.Lambda parseLambda(Lambda expr) {
	return new Interpreter.Lambda
	    (makeSymbolList(expr.getParameters()),
	     parseExpression(expr.getIn()));
    }

    private List makeSymbolList(ListOfName names) {
	List result = new LinkedList();
	for (Iterator i = names.iterator(); i.hasNext();) {
	    result.add(Symbol.intern(i.next().toString()));
	}
	return result;
    }

    Interpreter.While parseWhile(While expr) {
	return new Interpreter.While
	    (parseExpression(expr.getTest()),
	     parseExpression(expr.getRepeat()));
    }

}
