/* Generic wrapper for simple information source I-X agents.
 * Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar 18 17:24:24 2004 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.iquery;

import java.util.*;

import ix.iface.util.Reporting;

import ix.icore.*;
import ix.iface.util.ToolController;
import ix.util.*;
import ix.util.lisp.*;

/**
 * Generic base for simple information source I-X agents.  Such
 * an agent is typically defined by subclassing both this class and
 * {@link LookupHandler} and defining the three methods those
 * classes leave abstract:
 * <ul>
 * <li>{@link #isLookupActivity(Activity)}
 *     - identifies activities that should be given to a
 *       {@link LookupHandler}.
 * <li>{@link #makeLookupHandler(IQuery, Activity)}
 *     - creates a {@link LookupHandler} suitable for handling the
 *       activity.
 * <li>{@link LookupHandler#handleLookup()}
 *     - handles the activity that was supplied when the
 *       {@link LookupHandler} was created.
 * </ul>
 * Here is an outline example in which an inner class is used
 * for the {@link LookupHandler} subclass:
 * <pre>
 * public class Example extends IQuery {
 *
 *     public Example() {
 *         super();
 *     }
 *
 *     public static void main(String[] argv) {
 *         Util.printGreeting("I-Q Example");
 *         new Example().mainStartup(argv);
 *     }
 *
 *     protected boolean isLookupActivity(Activity activity) {
 *         return activity.getPattern().get(0) == ...;
 *     }
 *
 *     protected LookupHandler
 *               makeLookupHandler(IQuery queryAgent,
 *                                 Activity lookupActivity) {
 *         return new ExampleLookupHandler(queryAgent, lookupActivity);
 *     }
 *
 *
 *     static class ExampleLookupHandler extends LookupHandler {
 *
 *         ExampleLookupHandler(IQuery queryAgent, Activity lookupActivity) {
 *             super(queryAgent, lookupActivity);
 *         }
 *
 *         protected void handleLookup() {
 *             try {
 *                 ...
 *                 sendReport(ReportType.SUCCESS, "Finished -- ...");
 *             }
 *             catch (Exception e) {
 *                 Debug.displayException(e);
 *                 sendReport(ReportType.FAILURE,
 *                            "Problem: " + Debug.describeException(e));
 *             }
 *         }
 *
 *     }
 *
 * }
 * </pre>
 *
 * <p>By default, IQuery provides only a simple GUI: a frame containing
 * an "Exit" button and a text area used to display a transcript of the
 * messages sent and received.  A different GUI could be created by
 * overriding the {@link #startup()} or {@link #setupGUI()} methods.</p>
 *
 * <p>When the agent receives a message, a description of the message
 * is added to the transcript text area.  Then, if the message is an
 * {@link Activity} and {@link #isLookupActivity(Activity)} returns true,
 * the {@link #handleLookupActivity(Activity)} method is called.</p>
 *
 * <p>That method arranges for each request to be handled in a
 * separate {@link IQuery.LookupThread}.  The thread creates a new
 * {@link LookupHandler} by calling 
 * {@link #makeLookupHandler(IQuery, Activity)}
 * and then calls the handler's
 * {@link LookupHandler#handleLookup()} method.</p>
 *
 * <p>Most activities of an I-X agent take place in the AWT event-handling
 * thread, and operations on Swing GUI components should take place there
 * as well, so {@link LookupHandler#handleLookup()} has to take care
 * to do things in the right thread.  The {@link LookupHandler}
 * class provides some utility methods that deal with the most
 * common cases: adding text to the transcript, and sending a
 * report back to the agent that sent the lookup activity.
 * In other cases, {@link Util#swingAndWait(Runnable)} might
 * be used.  {@link Debug#displayException(Throwable)} can be used
 * in any thread to put up a dialog window that tells the user
 * of an exception.</p>
 *
 * <p>If for some reason you don't want a separate thread for
 * each request, override {@link #handleLookupActivity(Activity)}.</p>
 */
public abstract class IQuery extends IXAgent {

    public IQuery() {
	super();
    }

    /**
     * Command-line argument processing.
     */
    protected void processCommandLineArguments() {
	super.processCommandLineArguments();
    }

    /**
     * Agent setup and initialization.
     * The method supplied by the IQuery class just
     * calls {@link #setupGUI()} in the AWT event thread.
     *
     * @see IXAgent#startup()
     */
    protected void startup() {
	Util.swingAndWait(new Runnable() {
	    public void run() {
		setupGUI();
	    }
	});
    }

    /**
     * Creates a "transcript" frame for this agent, including
     * an "Exit" button.
     */
    protected void setupGUI() {
	// This is optional.  We could just let the first incoming message
	// cause the text frame to be created.  But this way we can make it
	// appear right away and can add an "Exit" button.

	textFrame = new TextAreaFrame("Messages for " + ipcName,
				      new String[] { "Exit" } );
	textFrame.setFoldLongLines(0);
	textFrame.addToolManager();
	textFrame.addListener(new TextAreaFrame.TListener() {
	    public void buttonPressed(String action) {
		if (action.equals("Exit"))
		    if (Util.dialogConfirms
			    (textFrame, "Are you sure you want to exit?"))
			exit();
	    }
	});
    }

    /**
     * Called when the agent should cease execution.
     */
    public void exit() {
	System.exit(0);
    }

    public void addTool(ToolController tc) {
	textFrame.addTool(tc);
    }

    /**
     * A utility that adds a line to this agent's transcript text area.
     * This method is normally called by other utility methods,
     * including ones in the {@link LookupHandler} class,
     * rather than being called directly.
     */
    public void do_transcript(String line) {
	textFrame.appendLine(line);
	if (!textFrame.getFrame().isShowing())
	    textFrame.setVisible(true);
    }

    /**
     * Handles new activities from external sources.
     * A description of the activity is added to the transcript text
     * area.  Then, if {@link #isLookupActivity(Activity)} returns true,
     * handleNewActivity calls {@link #handleLookupActivity(Activity)};
     * otherwise, the activity is ignored.
     */
    public void handleNewActivity(Activity activity) {
	handleReceivedReport(activity);
	do_transcript(Reporting.activityDescription(activity));
	if (isLookupActivity(activity))
	    handleLookupActivity(activity);
    }

    /**
     * Handles the activity be creating and starting a new
     * {@link IQuery.LookupThread}.
     */
    protected void handleLookupActivity(Activity activity) {
	new LookupThread(activity)
	    .start();
    }

    /**
     * A thread that invokes a {@link LookupHandler}.
     *
     * <p>When started, this thread makes a new {@link LookupHandler}
     * by calling {@link #makeLookupHandler(IQuery, Activity)} and then 
     * calls the handler's {@link LookupHandler#handleLookup()} method.
     */
    class LookupThread extends CatchingThread {
	Activity act;
	LookupThread(Activity act) { this.act = act; }
	public void innerRun() {
	    makeLookupHandler(IQuery.this, act)
		.handleLookup();
	}
    }

    /* Here's what a subclass must define. */

    /**
     * Determines whether the activity is one that should be handled
     * by a {@link LookupHandler}.
     *
     * @see #handleNewActivity(Activity)
     */
    protected abstract boolean isLookupActivity(Activity activity);

    /**
     * Return a {@link LookupHandler} for an activity
     * that was approved by {@link #isLookupActivity(Activity)}.
     */
    protected abstract LookupHandler
	      makeLookupHandler(IQuery queryAgent,
				Activity lookupActivity);

}

