/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Oct 19 17:34:29 2006 by Jeff Dalton
 * Copyright: (c) 1993, 1994, 2005-2006, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.io.PrintStream;

import java.util.*;

import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.process.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.context.*;

/**
 * Time-point Network.
 */
public class TimePointNet implements ConstraintManager {

    protected ContextMap knownPoints = new ContextHashMap(); // really a Set

    protected ContextInt numberOfPoints = new ContextInt(0);

    protected TimePoint timePointAtZero = null;

    /**
     * Constructs an empty time-point network.
     */
    public TimePointNet() {
    }

    protected int getNumberOfPoints() {
	return numberOfPoints.get();
    }

    public void clear() {
	// Remember that we don't have a list of all points or
	// of all constraints.
	numberOfPoints = new ContextInt(0);
	timePointAtZero = null;
    }

    /**
     * An internal representation of temporal "before" constraints.
     */
    public /* protected */ static class TimeConstraint {

	// /\/: This class s.b. protected rather than public,
	// but we refer to it in ix.icore.process.TimePoint.

	protected final TimePoint pre;
	protected final TimePoint post;
	protected final long min;
	protected final long max;

// 	public TimeConstraint(TimePoint pre, TimePoint post) {
// 	    this(pre, post, 0, Inf.INFINITY);
// 	}

	public TimeConstraint(TimePoint pre, TimePoint post,
			      long min, long max) {
	    this.pre = pre;
	    this.post = post;
	    this.min = min;
	    this.max = max;
	}

	public TimePoint getPrePoint() {
	    return pre;
	}

	public TimePoint getPostPoint() {
	    return post;
	}

	public long getMin() {
	    return min;
	}

	public long getMax() {
	    return max;
	}

	public String toString() {
	    return "TimeConstraint[" + pre + arrow() + post + "]";
	}

	private String arrow() {
	    return "-- " + min + ".." + Inf.asString(max) + " -->";
	}

    }

    /** @throws UnsupportedOperationException if called. */
    public void addConstraints(List constraints) {
	throw new UnsupportedOperationException
	    ("Method addConstraints(List) is not supported by TimePointNet.");
    }

    /** @throws UnsupportedOperationException if called. */
    public void addConstraints(PNode node, List constraints) {
	throw new UnsupportedOperationException("not ready");
    }

    public void addConstraint(PNode node, Constraint c) {
	// /\/: Handles only duration constraints for now.
	Debug.noteln("Adding time constraint", c);
	Debug.expectEquals("duration", c.getRelation().toString());
	Debug.expect(c.getParameters().size() == 2);
	Debug.expectEquals("self", c.getParameters().get(0).toString());
	TimePoint from = node.getBegin();
	TimePoint to = node.getEnd();
	TimeWindow w = (TimeWindow)c.getParameters().get(1);
	long min = w.getMin().asMilliseconds();
	long max = w.getMax() == null ? Inf.INFINITY 
	              : w.getMax().asMilliseconds();
	addTimePoint(from);
	addTimePoint(to);
	if (!addTimeConstraint(from, to, min, max))
	    throw new FailureException("Cannot add " + c);
	
    }

    public TimePoint addTimePoint(long min, long max) {
	// /\/: Given the way this is curently used -- only in tests --
	// it doesn't need to record the point as "known".
	Debug.expect(max == Inf.INFINITY || min <= max);
	numberOfPoints.add(1);
	return new TimePoint(min, max);
    }

    public TimePoint addTimePoint(TimePoint t) {
	if (knownPoints.get(t) != Boolean.TRUE) {
	    knownPoints.put(t, Boolean.TRUE);
	    numberOfPoints.add(1);
	}
	return t;
    }

    public void makeTimePointAtZero() {
	timePointAtZero = addTimePoint(0, 0);
    }

    public void addTimeConstraintElseFail(TimePoint pre, TimePoint post) {
	// /\/: This method doesn't "fit" with the others very well.
	// /\/: Should the addTimePoint calls be moved to addTimeConstraint?
	addTimePoint(pre);
	addTimePoint(post);
	if (!addTimeConstraint(pre, post, 0, Inf.INFINITY))
	    throw new FailureException
		("Cannot add " + pre + " -- 0..inf --> " + post);
    }

    public boolean addTimeConstraint(TimePoint pre, TimePoint post,
				     long min, long max) {
	Debug.expect(max == Inf.INFINITY || min <= max);
	TimeConstraint c = new TimeConstraint(pre, post, min, max);
	pre.addPostConstraint(c);
	post.addPreConstraint(c);
	if (propagateAfterAdding(c)) {
	    Debug.noteln("Added", c);
	    return true;
	}
	else {
	    deleteTimeConstraint(c);
	    Debug.noteln("Failed to add", c);
	    return false;
	}
    }

    /** Removes the constraint from its pre- and post-points. */
    protected void deleteTimeConstraint(TimeConstraint c) {
	c.getPrePoint().removePostConstraint(c);
	c.getPostPoint().removePreConstraint(c);
    }

    public void describeTPN(PrintStream out) {
	out.println("Time Point Net");
	for (Iterator pi = knownPoints.keySet().iterator(); pi.hasNext();) {
	    TimePoint p = (TimePoint)pi.next();
	    out.println(p);
	    for (Iterator ci = p.getPostConstraints().iterator()
		     ; ci.hasNext();) {
		TimeConstraint c = (TimeConstraint)ci.next();
		out.println("   " + c);
	    }
	    out.println("");
	}
    }

    protected boolean propagateAfterAdding(TimeConstraint c) {
	TimePoint pre = c.getPrePoint();
	TimePoint post = c.getPostPoint();
	long cmin = c.getMin();
	long cmax = c.getMax();
	long preMin = pre.getMinTime();
	long postMin = post.getMinTime();
	long preMax = pre.getMaxTime();
	long postMax = post.getMaxTime();
	// If nothing will change, we can just return true now.
	if (preMin + cmin > postMin          // post min pushed up
	    || (cmax != Inf.INFINITY
		&& postMin - cmax > preMin)  // pre min pulled up
	    || Inf.isLess(Inf.diff(postMax, cmin),
			  preMax)            // pre max pushed down
	    || Inf.isLess(Inf.add(preMax, cmax),
			  postMax))          // post max pulled down
	    return updateTimeWindows
		     (Lisp.list(pre, post));
	else
	    return true;
    }

    private int mark = 0;
    private int baseMark = 0;
    private List changeHistory;
    private List changedPoints;

    protected boolean updateTimeWindows(List initialPoints) {
	try {
	    changeHistory = new LinkedList();
	    if (findMinValues(initialPoints)) {
		if (findMaxValues(initialPoints))
		    // Successfully updated the net.
		    return true;
		else
		    // This should never happen.
		    throw new ConsistencyException("Can't find max values.");
	    }
	    else {
		// The net has inconsistent constraints and is not invalid.
		// We have to undo all our changes and put it back in its
		// earlier, valid state.
		undoTpnChanges(changeHistory);
		return false;
	    }
	}
	finally {
	    changeHistory = null;
	    changedPoints = null;
	}
    }

    boolean findMinValues(List initialActivePoints) {
	List activePoints = initialActivePoints; // check this iteration
	changedPoints = new LinkedList();        // check next iteration
	int n = getNumberOfPoints(); // number of points
	int iterations = 0;          // number of iterations completed
	baseMark = mark;
	// Repeatedly process constraint arcs from the active points to
	// get a new list of active points.  The min of a point is the
	// length of the longest path (so far) from an imaginary point-0
	// at 0.  The max values of constraints and points are interpreted
	// as backward arcs of negated length.
	// Allow n+1 iterations.
	while (!activePoints.isEmpty() && iterations <= n) {
	    // Each iteration needs a new change-mark.
	    mark = newMark();
	    // Process all arcs by visiting all points, collecting
	    // a list of changed-points as we go.  The changed points
	    // will become the active points for the next iteration.
	    for (Iterator i = activePoints.iterator(); i.hasNext();) {
		TimePoint tp = (TimePoint)i.next();
		// Process arc from tp back to the imaginary point-0.
		// This would make the distance to point-0 be < 0 when
		// the min of the point is > the max.  So in that case
		// we exit, indicating failure.
		if (!(tp.getMaxTime() == Inf.INFINITY
		      || tp.getMaxTime() >= tp.getMinTime()))
		    return false;
		// Process forward arcs to successors.  This may push up
		// the min of some successors.
		for (Iterator ci = tp.getPostConstraints().iterator()
			 ; ci.hasNext();) {
		    TimeConstraint c = (TimeConstraint)ci.next();
		    Debug.expectSame(tp, c.getPrePoint());
		    constrainMin(tp, c.getPostPoint(), c.getMin());
		}
		// Process backward arcs to predecessors, negating length.
		// This may pull up the min of some predecessors.
		for (Iterator ci = tp.getPreConstraints().iterator()
			 ; ci.hasNext();) {
		    TimeConstraint c = (TimeConstraint)ci.next();
		    Debug.expectSame(tp, c.getPostPoint());
		    constrainMin(tp, c.getPrePoint(), -c.getMax());
		}
	    }
	    activePoints = changedPoints;
	    changedPoints = new LinkedList();
	    iterations++;
	}
	// Converged if no points changed on the last iteration.
	return activePoints.isEmpty();
    }

    /* Processing arcs of length len from pre to succ */

    void constrainMin(TimePoint pre, TimePoint post, long len) {
	// If the distance to the successor is less than the distance
	// to the predecessor plus the arc length, make it be the
	// distance to the predecessor plus the arc length.
	long distSum = pre.getMinTime() + len;
	if (distSum > post.getMinTime()) {
	    if (!(post.tpn_mark > baseMark))
		changeHistory.add(new Change(post));
	    post.setMinTime(distSum);
	    if (!(post.tpn_mark == mark)) {
		post.tpn_mark = mark;
		changedPoints.add(post);
	    }
	}
    }

    /* Undoing changes */

    void undoTpnChanges(List changeHistory) {
	for (Iterator i = changeHistory.iterator(); i.hasNext();) {
	    ((Change)i.next()).undo();
	}
    }

    static class Change {
	TimePoint point;
	long oldMin;
	Change(TimePoint p) {
	    point = p; oldMin = p.getMinTime();
	}
	void undo() {
	    point.setMinTime(oldMin);
	}
    }

    /* Marking */

    private int newMark() {
	// use: mark = newMark();
	Debug.expect(mark < Integer.MAX_VALUE);
	return mark + 1;
    }

    /* Finding max values */

    boolean findMaxValues(List initialActivePoints) {
	List activePoints = initialActivePoints; // check this iteration
	changedPoints = new LinkedList();        // check next iteration
	int n = getNumberOfPoints(); // number of points
	int iterations = 0;          // number of iterations completed
	while (!activePoints.isEmpty()) {
	    Debug.expect(iterations < n);
	    mark = newMark();
	    for (Iterator i = activePoints.iterator(); i.hasNext();) {
		TimePoint tp = (TimePoint)i.next();
		// Process arcs forward from predecessors.  This may push down
		// the max of a predecessor to be the min distance before the
		// max of tp.
		for (Iterator ci = tp.getPreConstraints().iterator()
			 ; ci.hasNext();) {
		    TimeConstraint c = (TimeConstraint)ci.next();
		    Debug.expectSame(tp, c.getPostPoint());
		    constrainMax(c.getPrePoint(), tp, c.getMin());
		}
		// Process backward arcs from successors, negating length.
		// This may pull down the max of some successors.
		for (Iterator ci = tp.getPostConstraints().iterator()
			 ; ci.hasNext();) {
		    TimeConstraint c = (TimeConstraint)ci.next();
		    Debug.expectSame(tp, c.getPrePoint());
		    if (c.getMax() != Inf.INFINITY)
			constrainMax(c.getPostPoint(), tp, -c.getMax());
		}
	    }
	    activePoints = changedPoints;
	    changedPoints = new LinkedList();
	    iterations++;
	}
	return true;
    }

    void constrainMax(TimePoint pre, TimePoint post, long len) {
	// If the value at the predecessor is greater than the value at
	// the successor minus the arc length, make it be the value at the
	// successor minus the arc length.
	if (post.getMaxTime() != Inf.INFINITY) {
	    long newLow = post.getMaxTime() - len;
	    long preMax = pre.getMaxTime();
	    if (preMax == Inf.INFINITY || newLow < preMax) {
		pre.setMaxTime(newLow);
		if (!(pre.tpn_mark == mark)) {
		    pre.tpn_mark = mark;
		    changedPoints.add(pre);
		}
	    }
	}
    }

}
