/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed May 31 02:39:53 2006 by Jeff Dalton
 * Copyright: (c) 2004, 2005, 2006, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.io.PrintStream;

import java.util.*;

import ix.icore.process.PNodeEnd;
import ix.util.*;

/**
 * Gathers or caculates various statistics for the Slip algorithm.
 */
public class SlipStats implements PlanStats {

    Slip slip;
    boolean filledInValues = false;
    int numberStepsTaken = 0;
    int numberAltsPosted = 0;
    int numberAltsPicked = 0;
    int numberAltsLeft = -1;
    int numberNodes = -1;
    int longestPathLength = -1;
    Map otherStats = null;

    public SlipStats(Slip slip) {
	this.slip = slip;
    }

    public void stepTaken() {
	numberStepsTaken++;
    }

    public void altPosted() {
	numberAltsPosted++;
    }

    public void altPicked() {
	numberAltsPicked++;
    }

    public void recordStat(String key, Object value) {
	if (otherStats == null)
	    otherStats = new TreeMap();
	otherStats.put(key, value);
    }

    public void incrementStat(String key) {
	if (otherStats != null) {
	    Integer count = (Integer)otherStats.get(key);
	    if (count != null) {
		otherStats.put(key, new Integer(count.intValue() + 1));
		return;
	    }
	}
	recordStat(key, new Integer(1));
    }

    public void addStats(PlanStats planStats) {
	SlipStats stats = (SlipStats)planStats;
	if (!filledInValues) {
	    numberAltsLeft = 0;
	    numberNodes = 0;
	    longestPathLength = 0;
	    filledInValues = true;
	}
	numberStepsTaken += stats.numberStepsTaken;
	numberAltsPosted += stats.numberAltsPosted;
	numberAltsPicked += stats.numberAltsPicked;
	// Since the alts left stay around to be used in subsequent
	// replans, they'd be counted more than once if we just
	// summed the numberAltsLeft values.
	numberAltsLeft += stats.numberAltsPosted - stats.numberAltsPicked;
	numberNodes += stats.numberNodes;
	longestPathLength += stats.longestPathLength;
	if (stats.otherStats != null) {
	    for (Iterator i = stats.otherStats.entrySet().iterator()
		     ; i.hasNext();) {
		Map.Entry e = (Map.Entry)i.next();
		String k = (String)e.getKey();
		Integer v = (Integer)e.getValue();
		Integer n = (Integer)otherStats.get(k);
		if (n == null)
		    otherStats.put(k, v);
		else
		    otherStats.put(k, new Integer(n.intValue()
						  + v.intValue()));
	    }
	}
    }

    /**
     * Calculates and stores the statistics that are not accumulated
     * during planning but instead computed after planning has finished.
     */
    void fillInValues() {
	numberAltsLeft = slip.alternatives.size();
	numberNodes = slip.MM().getNodes().size();
	longestPathLength = calcLongestPathLength();
	filledInValues = true;
    }

    public void report(PrintStream out) {
	Debug.expect(filledInValues, "missing values for stats");
	out.println("Planning statistics:");
	out.println("Steps taken = " + numberStepsTaken);
	out.println("Alternatives posted = " + numberAltsPosted);
	out.println("Alternatives picked = " + numberAltsPicked);
	out.println("Alternatives remaining = " + numberAltsLeft);
	out.println("Number of nodes = " + numberNodes);
	out.println("Longest node-end path length = " + longestPathLength);
	if (otherStats != null) {
	    for (Iterator i = otherStats.entrySet().iterator(); i.hasNext();) {
		Map.Entry e = (Map.Entry)i.next();
		out.println(e.getKey() + " = " + e.getValue());
	    }
	}
    }

    public int calcLongestPathLength() {
	// Get a list of all the node-ends.
	return calcLongestPathLength(slip.MM().getNodeEnds());
    }

    public int calcLongestPathLength(final List nodeEnds) {
	// Make a DirectedGraph from the node-ends and their successor links.
	DirectedGraph g = new AbstractDirectedGraph() {
	    public Collection getAllNodes() { return nodeEnds; }
	    public Collection getRoots() { return nodeEnds; } // /\/
	    public Collection getSuccessors(Object nodeEnd) {
		return ((PNodeEnd)nodeEnd).getSuccessors();
	    }
	    public String toString() {
		return "DirectedGraph[" + getAllNodes() + "]";
	    }
	};
	// Add unique start and finish nodes.
	BoundedGraph bounded = Graphs.makeBoundedGraph(g);
	// Calculate all path lengths.
	Map distances = Graphs.findLongestPathLengths(bounded);
	// Get length of longest path to the finish node.
	IntRef len = (IntRef)distances.get(bounded.getFinishNode());
	// We don't are about the initial link from the start node
	// or the final link to the finish node, so subtract 2.
	return len.get() - 2;
    }

}
