/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Feb 16 02:51:48 2005 by Jeff Dalton
 * Copyright: (c) 2004, 2005, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.context.*;

/**
 * A single-stepper for an instance of {@link Slip}.
 */
public class SlipSingleStepper {

    Slip slip;

    Map commandTable = new HashMap();

    Trigger trigger;

    public SlipSingleStepper(Slip slip) {
	this.slip = slip;
	this.trigger = instantTrigger;
    }

    /**
     * Called before each planning step when single-stepping is on.
     */
    public void step() {
	if (trigger.isTriggered()) {
	    trigger = instantTrigger;
	    singleStepBreakLoop();
	}
    }

    void singleStepBreakLoop() {
	SlipStats stats = (SlipStats)slip.getStatistics();
	String prompt = "Step " + (stats.numberStepsTaken + 1) + ":";
	while (true) {
	    String line = Util.askLine(prompt);
	    LList command = Lisp.elementsFromString(line);
	    if (command.isEmpty())
		return;
	    else
		evalCommand(command);
	}
    }

    /* * * Triggers * * */

    abstract class Trigger {
	abstract boolean isTriggered();
    }

    Trigger instantTrigger = new Trigger() {
	boolean isTriggered() { return true; }
    };

    /* * * Commands * * */

    void evalCommand(LList command) {
	String name = command.car().toString();
	Command c = (Command)commandTable.get(name);
	if (c == null)
	    Debug.noteln("I don't know how to", name);
	else
	    c.run(command.cdr());
    }

    void define(Command c) {
	commandTable.put(c.name, c);
    }

    abstract class Command {
	String name;
	Command(String name) {
	    this.name = name;
	}
	abstract void run(LList args);
    }

    String[] helpText = {
        "nodes",
	"alts",
	"contexts",
	"help"
    };

    /** Command definitions */
    {

	define(new Command("nodes") {
	    void run(LList args) {
		slip.describeNodeEnds();
	    }
	});

	define(new Command("alts") {
	    void run(LList args) {
		slip.describeAlternatives();
	    }
	});

	define(new Command("contexts") {
	    void run(LList args) {
		Context.printContextTree();
	    }
	});

	define(new Command("help") {
	    void run(LList args) {
		Util.printLines(helpText);
	    }
	});

    }

}

// Issues:
// * Debug.out might not be System.out.
