/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Apr  7 23:56:20 2005 by Jeff Dalton
 * Copyright: (c) 2004, 2005, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.ip2.*;
import ix.icore.*;
import ix.icore.process.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

// /\/: Still requires effects to be only at END ends.
// /\/: Conditions only at BEGIN ends.
// /\/: Assumes all conds and effects come from refinements that put them
//      at BEGIN or END of self during planning.  So if the planner doesn't
//      expand a node, it assumes the node has no conditions.

/**
 * Finds an executable node-end and either simulates its execution
 * or begins its expansion.
 */
class SlipFindExecutable implements Runnable {

    Slip slip;

    SlipFindExecutable(Slip slip) {
	this.slip = slip;
    }

    public void run() {
	LList nodeEnds = getNodeEnds();
	findNext(nodeEnds, endIterator(nodeEnds));
    }

    void findNext(LList allNodeEnds, LListIterator nodeEndIter) {
	while (true) {
	    // First, find a node-end with status POSSIBLE.
	    // If none exists, an exception is thrown to indicate
	    // either that the plan is complete or that we must
	    // try an alternative.
	    PNodeEnd ne = nextStatusExecutable(nodeEndIter);
	    Debug.noteln("SlipFindExecutable considering", ne);
	    // END ends don't expand, but they may execute.
	    if (ne.getEnd() == End.END) {
		// Could the node-end execute?
		if (canComplete(ne)) {
		    // Yes -- we'll execute this node-end after posting
		    // an alt that will try others instead.
		    postFindNextAlt(ne, allNodeEnds, nodeEndIter);
		    // Process its effects and make it COMPLETE.
		    executeNodeEnd(ne);
		    // More node-ends might now have status POSSIBLE.
		    nodeEndIter = endIterator(allNodeEnds);
		}
		continue;
	    }
	    // (From here on, we know we have a BEGIN end.)
	    PNode node = ne.getNode();
	    // It might be a goal node
	    if (slip.isGoalNode(ne)) {
		Debug.noteln("Examining goal node", ne);
		PatternAssignment goalCond = slip.getGoalCond(ne);
		ListOfConstraint goalList = new LinkedListOfConstraint();
		goalList.add(makeCondition(goalCond));
		Debug.noteln("Goal cond", goalCond);
		List envs = slip.MM().evalFilters(goalList, new MatchEnv());
		// If the cond can be satisfied, we delete the goal node.
		if (!envs.isEmpty()) {
		    Debug.noteln("Envs for goal cond", envs);
		    // We might have tried a different node-end first.
		    postFindNextAlt(ne, allNodeEnds, nodeEndIter);
		    // Tell the variable manager about the ways the
		    // cond might be satisfied.
		    // /\/: We ought to protect over the range from
		    // the possible contributors to the node-end that
		    // wants the goal cond achieved.  That would need
		    // a multiple-contributor GOST entry.
		    List branches = slip.MM().reevaluateFilters(goalList);
		    Debug.expect(!branches.isEmpty(),
				 "Cannot match goal", goalCond);
		    // Also, the node that had the condition might
		    // now be executable, since it no longer needs
		    // to wait for the goal node.  /\/: Since we don't
		    // have a handy ref to that node-end, we'll make
		    // the end of the goal complete before deleting,
		    // which should do the trick.
		    slip.setStatus(node.getEnd(), Status.COMPLETE);
		    // Delete the goal node.
		    ActivityItem item = (ActivityItem)ne.getNode();
		    Agenda actAgenda = slip.getController()
			.getActivityAgenda();
		    Debug.noteln("Deleting goal node", item);
		    actAgenda.removeItem(item);
		    // Since we've changed the list of nodes, we
		    // need to get it again.
		    allNodeEnds = getNodeEnds();		    
		    // Find the next node-end to execute.
		    // We (probably?) don't make anything executable except
		    // the node-end that wanted the condition satisfied,
                    // but we may be in an alt in which we skipped an
		    // executable earlier in the list of node-ends.
		    nodeEndIter = endIterator(allNodeEnds);
		    continue;
		}
		// Otherwise, treat as if it were a normal node.
		Debug.expect(!node.isExpanded());
	    }
	    // The node may already be expanded.
	    if (isAlreadyExpanded(ne)) {
		Debug.noteln("Is already expanded");
		ListOfConstraint conds = slip.MM().getNodeConditions(node);
		if (conds == null || conds.isEmpty()) {
		    // No conditions, so treat as primitive
		    Debug.noteln("No conditions");
		    postFindNextAlt(ne, allNodeEnds, nodeEndIter);
		    slip.setStatus(ne, Status.COMPLETE);
		    nodeEndIter = endIterator(allNodeEnds);
		    continue;
		}
		List envs = slip.MM().evalFilters(conds, new MatchEnv());
		Debug.noteln("Conditions", conds);
		Debug.noteln("Envs", envs);
		// The envs tell us all the ways we could satisfy
		// all the filters by binding.  Each env is a way to
		// satisy all the filters.
		if (envs.isEmpty()) {
		    // Cannot satisfy the conds, so not really executable
		    // at this point.
		    Debug.noteln("Cannot satisfy conds");
		    // If it looks like we might be able to satisfy the
		    // conds by introducing some nodes, post an alt that
		    // does that.  /\/: Perhaps we should do this even
		    // if all conds can be satisfied now.
		    if (!slip.isWaitingForAchieve(node)
			  && mightAchieveConds(conds)) {
			postSatisfyByAchieveAlt(node, conds);
		    }
		    // Try a different node-end now, because we can't
		    // execute this one.
		    continue;
		}
		// We can satisfy the conditions.
		// We're going to do something with this node-end,
		// so post an alt to try another instead.
		postFindNextAlt(ne, allNodeEnds, nodeEndIter);
		// Satisfy the conds, posting an alt for any other
		// ways to satisfy them.
		satisfyNodeEndConds(ne, conds, envs);
		// See what node-ends are now POSSIBLE.
		nodeEndIter = endIterator(allNodeEnds);
		continue;
	    }
	    // Try refinements
	    Debug.noteln("Trying refinements");

	    LinkedList useNow = new LinkedList();
	    LinkedList useLater = new LinkedList();
	    if (!nodeExpanders(ne, useNow, useLater)) {
		// No refinements apply, so treat as primitive.
		Debug.noteln("No refinements apply");
		postFindNextAlt(ne, allNodeEnds, nodeEndIter);
		slip.setStatus(ne, Status.COMPLETE);
		nodeEndIter = endIterator(allNodeEnds);
		continue;
	    }
	    if (!useNow.isEmpty()) {
		// The node can be expanded now.
		Debug.noteln("Can expand");
		// Though we might have picked another node instead.
		postFindNextAlt(ne, allNodeEnds, nodeEndIter);
		if (!useLater.isEmpty()) {
		    // We can also expand it later
		    postExpandLaterAlt(node, useLater);
		}
		// Do the expansion.
		slip.setNextStep(new SlipExpandNode(slip, node, useNow));
		return;
	    }
	    if (!useLater.isEmpty()) {
		// If it looks like we might be able to expand if we
		// introduced some nodes to satisfy conditions,
		// post an alt that does that.  /\/: We might want
		// to allow node-introduction even in the useNow case
		// above.  [Done]
		postExpandLaterAlt(node, useLater);
		// But continue looking for a node that can be
		// executed now.
	    }
	    // At least one refinement applied, but none could be used now.
	    // So try to exec something else.
	    Debug.noteln("No refinement can expand now");
	    continue;
	}
    }

    void postFindNextAlt(PNodeEnd selected,
			 LList allNodeEnds,
			 LListIterator nodeEndIter) {
	// An alt is posted only if there is another executable node-end.
	// Otherwise, the alt, if chosen, would just poison without
	// doing anything interesting.
	if (nodeEndIter.hasNext()) {
	    // Make a new iterator to avoid modifying the existing one.
	    // This is a precaution -- it's not clear it's necessary, /\/
	    // but we need a ListIterator in any case, so that we can
	    // go back 1.
	    LListListIterator i =
		new LListListIterator(nodeEndIter.getTail());
	    // See if there's another executable node-end.
	    PNodeEnd ne = slip.findExecutable(i);
	    if (ne != null) {
		// There is another executable.  Unfortunately, our
		// iterator has now moved one step too far.
		i.previous();
		slip.postAlternative(new Resume(allNodeEnds, i));
	    }
	}
    }

    class Resume extends Alternative {
	LList allNodeEnds;
	LListIterator nodeEndIter;

	Resume(LList allNodeEnds, LListIterator nodeEndIter) {
	    this.allNodeEnds = allNodeEnds;
	    this.nodeEndIter = nodeEndIter;
	}

	public void run() {
	    findNext(allNodeEnds, nodeEndIter);
	}

    }

    /*
     * Condition satisfaction
     */

    Constraint makeCondition(PatternAssignment pv) {
	return new Constraint
	    (Refinement.S_WORLD_STATE, Refinement.S_CONDITION,
	     Lisp.list(pv));
    }

    void satisfyNodeEndConds(PNodeEnd ne, ListOfConstraint conds, List envs) {
	AgendaItem item = (AgendaItem)ne.getNode();
	if (envs.size() == 1) {
	    // Only one way to satisfy the conds, so we can just
	    // do it.
	    slip.MM().satisfyConds(item, conds, (MatchEnv)envs.get(0));
	    slip.setStatus(ne, Status.COMPLETE);
	    return;
	}
	// Multiple envs.
	// /\/: Here we rely on the envs being a LinkedList so that
	// we can use them as a queue in alternatives.
	LinkedList condEnvs = (LinkedList)envs;
	MatchEnv e1 = (MatchEnv)condEnvs.removeFirst();
	slip.postAlternative
	    (new SlipSatisfyConds(slip)
	     .new ResumeSatisfaction
	            (item, conds, condEnvs));
	slip.MM().satisfyConds(item, conds, e1);
	slip.setStatus(ne, Status.COMPLETE);
    }

    void postSatisfyByAchieveAlt(PNode node, ListOfConstraint conds) {
	slip.postAlternative(new SatisfyByAchieveAlt(node, conds));
    }

    class SatisfyByAchieveAlt extends Alternative {

	PNode node;
	ListOfConstraint conds;

	SatisfyByAchieveAlt(PNode node, ListOfConstraint conds) {
	    this.node = node;
	    this.conds = conds;
	}

	public void run() {
	    slip.setNextStep(new SlipAchieveConds(slip, node, conds));
	}

    }

    /**
     * Determines whether all the conditions can be satisfied
     * individually either by the current world-state or by
     * introducing nodes.  It does not determine whether all
     * the conds can be satisfied together; nor does it determine
     * that introducing nodes will definitely work, because it
     * might not be possible to get the right variable bindings.
     * At least one of the conditions must look like it can be
     * solved by introducing a node; for now at least /\/ this method
     * is called only when it's already known that the world state
     * cannot satisfy all of the conds together.
     */
    boolean mightAchieveConds(ListOfConstraint conds) {
	boolean haveAnAchievable = false;
	for (Iterator ci = conds.iterator(); ci.hasNext();) {
	    Constraint c = (Constraint)ci.next();
	    if (c.getType() != Refinement.S_WORLD_STATE)
		continue;
	    PatternAssignment pv = c.getPatternAssignment();
	    if (condMightBeAchieved(pv))
		haveAnAchievable = true;
	    else if (!worldStateSatisfiesCond(c))
		return false;
	}
	return haveAnAchievable;
    }

    boolean worldStateSatisfiesCond(Constraint cond) {
	//\/ Inefficient but simple.
	ListOfConstraint conds = new LinkedListOfConstraint();
	conds.add(cond);
	List envs = slip.MM().evalFilters(conds, new MatchEnv());
	return !envs.isEmpty();
    }

    boolean condMightBeAchieved(PatternAssignment cond) {
	return slip.existsAchieversForCond(cond);
    }

    /*
     * Node-end utilities
     */

    LList getNodeEnds() {
	// /\/: We use an LList because we know we can copy iterators.
	return LList.newLList(slip.MM().getNodeEnds());
    }

    LListIterator endIterator(LList l) {
	// This is called to get the initial nodeEndIter
	// when we want to find an executable.  We can pre-process
	// the list to execute any "trivial" node-ends -- ones
	// that have neither conditions nor effects -- because
	// it doesn't matter when they're executed relative to
	// other node-ends.  This reduces the number of permutations
	// that might be considered, because the trivial ends
	// aren't permuted.
	if (!slip.domainSays("iplan-try-pointless-permutations"))
	    executeTrivialEnds(l);
	return (LListIterator)l.iterator();
    }

    void executeTrivialEnds(LList nodeEnds) {
	boolean changed = true;
	while (changed) {
	    changed = false;
	    for (Iterator i = nodeEnds.iterator(); i.hasNext();) {
		PNodeEnd ne = (PNodeEnd)i.next();
		if (slip.getStatus(ne) == Status.POSSIBLE && isTrivial(ne)) {
		    Debug.noteln("Pre-executing", ne);
		    slip.setStatus(ne, Status.COMPLETE);
		    changed = true;
		}
	    }
	}
    }

    boolean isTrivial(PNodeEnd ne) {
	PNode node = ne.getNode();
	if (ne.getEnd() == End.BEGIN) {
	    ListOfConstraint conditions = slip.MM().getNodeConditions(node);
	    return (conditions == null || conditions.isEmpty())
		&& !slip.isGoalNode(ne)
		&& !someRefinementApplies(node);
	}
	else {
	    List effects = slip.MM().getNodeEffects(node);
	    return effects == null || effects.isEmpty();
	}
    }

    boolean someRefinementApplies(PNode node) {
	LList pattern = node.getPattern();
	for (Iterator i = slip.getDomain().getRefinements().iterator()
		 ; i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    if (refinementApplies(r, pattern))
		return true;
	}
	return false;
    }

    boolean refinementApplies(Refinement r, LList pattern) {
	// Determines whether the refinement is intended for
	// expanding activities with patterns like the one given.
	// This is not completely well-defined.  /\/
	LList refpat = r.getPattern();
	if (!refpat.car().equals(pattern.car()))
	    return false;
	int rlen = refpat.length(), plen = pattern.length();
	return rlen == plen
	    || (rlen-2 <= plen && refpat.contains(Matcher.REST));
    }

    PNodeEnd nextStatusExecutable(Iterator i) {
	PNodeEnd ne = slip.findExecutable(i);
	if (ne == null) {
	    if (slip.planIsComplete())
		throw new Slip.HavePlanException();
	    else
		throw slip.poison("no executable node-end");
	}
	return ne;
    }

    boolean canComplete(PNodeEnd ne) {
	PNode node = ne.getNode();
	List effects = slip.MM().getNodeEffects(node);
	if (effects == null)
	    return true;
	Set vars = Variable.varsAnywhereIn(effects);
	Set unbound = Variable.unboundVarsIn(vars);
	return unbound.isEmpty();
    }

    void executeNodeEnd(PNodeEnd ne) {
	PNode node = ne.getNode();
	List effects = slip.MM().getNodeEffects(node);
	if (effects != null)
	    slip.MM().handleEffects(node, effects);
	slip.setStatus(ne, Status.COMPLETE);
    }

    boolean isAlreadyExpanded(PNodeEnd ne) {
	Debug.expectSame(End.BEGIN, ne.getEnd());
	PNode node = ne.getNode();
	if (node.isExpanded()) {
	    Debug.expect(!slip.isGoalNode(ne));
	    return true;
	}
	else {
	    // If not expanded, there should (for now /\/) be
	    // no childred or conditions.
	    Debug.expect(node.getChildren().isEmpty(),
			 "Has children when not expanded", node);
	    Debug.expect(Collect.isEmpty(slip.MM().getNodeConditions(node)),
			 "Has conds when not expanded", node);
	    return false;
	}
    }

    /*
     * Expansion
     */

    boolean nodeExpanders(PNodeEnd ne, LinkedList useNow,
			               LinkedList useLater) {
	// Uses LinkedLists because they can be used like queues
	// in SlipExpandNode.
	// Returns false when no refinements apply, which means the
	// node needn't be expanded and can be treated as a primitive.
	if (slip.isGoalNode(ne))
	    return goalExpanders(ne, useNow, useLater);
	AgendaItem item = (AgendaItem)ne.getNode();
	boolean unusableRefinementApplies = false;
	AdviceManager advice = slip.MM().getAdviceManager();
	for (Iterator i = slip.getDomain().getRefinements().iterator()
		 ; i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    LList itemPat = item.getPattern();
	    // See if the refinement's pattern matches the node-end's.
	    // Advice may say a refinement cannot be used even though
	    // it matches.
	    MatchEnv e = Matcher.match(r.getPattern(), itemPat);
	    if (e == null || !advice.isUsableRefinement(r)) {
		// Refinement doesn't match, but perhaps it "applies".
		unusableRefinementApplies = unusableRefinementApplies
		    || refinementApplies(r, itemPat);
		continue;
	    }
	    if (Collect.isEmpty(r.getNodes())
		     && Collect.isEmpty(r.getConstraints())) {
		// When there are no subnodes or constraints, the
		// refinement name still specifies a way of doing
		// it manually.
		useNow.add(makeManualExpander(item, r, e));
	    }
	    else {
		// See if the refinement can be used now.
		Object expander = makeExpander(item, r, e);
		if (expander != null) {
		    // Add action to expand using the refinement.
		    useNow.add(expander);
		    continue;
		}
		// See if the refinement might be used if some nodes
		// were added to satisfy some conditions.
		expander = makeLaterExpander(item, r, e);
		if (expander != null) {
		    useLater.add(expander);
		    continue;
		}
		// The refinement matched the node's pattern, and
		// therefore applies, but cannot be used.
		unusableRefinementApplies = true;
	    }
	}
	return unusableRefinementApplies ||
	       !useNow.isEmpty() || !useLater.isEmpty();
    }

    boolean goalExpanders(PNodeEnd ne, LinkedList useNow,
			               LinkedList useLater) {
	// Always returns true (meaning some refinements apply),
        // because we wouldn't have created the goal node if
	// none applied.  ["Applies" here means that the
	// refinement has a matching effect, so it's stricter
	// than for expansion.]  We also know the refinement
	// must have at least one effect, so we don't have to
	// consider the "manual expander" case.
	AgendaItem item = (AgendaItem)ne.getNode();
	List achievers = slip.getGoalAchievers(item);
	for (Iterator i = achievers.iterator(); i.hasNext();) {
	    SlipAchiever achiever = (SlipAchiever)i.next();
	    Refinement r = achiever.refinement;
	    Debug.expect(!Collect.isEmpty(r.getConstraints()));

	    // Some Variables may have been bound since we
	    // matched the condition and effect.
	    MatchEnv e = achiever.rematchCond();
	    if (e == null) {
		Debug.noteln("Can't use " + achiever +
			     " for goal " + item +
			     " because " + achiever.condMatchEnv +
			     " no longer applies");
		continue;
	    }

	    // See if the refinement can be used now.
	    Object expander = makeExpander(item, r, e);
	    if (expander != null) {
		// Add action to expand using the refinement.
		useNow.add(expander);
		continue;
	    }
	    // See if the refinement might be used if some nodes
	    // were added to satisfy some conditions.
	    expander = makeLaterExpander(item, r, e);
	    if (expander != null) {
		useLater.add(expander);
		continue;
	    }
	    // The refinement cannot be used.
	    Debug.noteln("Can't use " + achiever + " for goal " + item);
	}
	if (useNow.isEmpty() && useLater.isEmpty())
	    Debug.noteln("Cannot expand goal", item);
	return true;
    }

    Object makeManualExpander(AgendaItem item, Refinement r, MatchEnv env) {
	// We're using filterEnvs == null to say the filters could not
	// be satisfied, so when they're satisfied trivially, because
	// there weren't any, we need to list the env from matching
	// the expands pattern.
	List filterEnvs = Lisp.list(env);
	return new SlipExpander(slip, item, r, env, filterEnvs);
    }

    Object makeExpander(AgendaItem item, Refinement r, MatchEnv env) {
	MatchEnv refinementMatchEnv = env;
	ListOfConstraint filters = r.getFilterConstraints();
	List filterEnvs = filters.isEmpty()
	    ? Collections.EMPTY_LIST
	    : slip.MM().evalFilters(filters, refinementMatchEnv);
	boolean isReady = filters.isEmpty() || !filterEnvs.isEmpty();
	if (isReady)
	    return new SlipExpander(slip, item, r,
				    refinementMatchEnv, filterEnvs);
	else
	    return null;
    }

    Object makeLaterExpander(AgendaItem item, Refinement r, MatchEnv env) {
	// This is called when there's no way to satisfy all of the
	// refinement's filter conds from the current world-state.
	// However, it still might be possible to satisfy them
	// individually or by adding nodes.
	if (!slip.isWaitingForAchieve(item) // ensure not already achieving
	      && mightAchieveConds(r.getFilterConstraints()))
	    return new SlipExpander(slip, item, r, env, null);
	else
	    return null;
    }

    void postExpandLaterAlt(PNode node, LinkedList useLater) {
	slip.postAlternative(new ExpandLaterAlt(node, useLater));
    }

    class ExpandLaterAlt extends Alternative {

	PNode node;
	LinkedList useLater;

	ExpandLaterAlt(PNode node, LinkedList useLater) {
	    this.node = node;
	    this.useLater = useLater;
	}

	public void run() {
	    slip.setNextStep(new SlipExpandNode(slip, node, useLater));
	}

    }

}
