/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Oct 19 17:54:53 2006 by Jeff Dalton
 * Copyright: (c) 2004 - 2006, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.ip2.*;
import ix.icore.*;
import ix.icore.process.*;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

/**
 * Represents one Refinement's way of expanding a node.
 */
class SlipExpander {

    Slip slip;
    AgendaItem item;
    Refinement refinement;
    MatchEnv refinementMatchEnv;
    List filterEnvs;

    SlipExpander(Slip slip, AgendaItem item, Refinement r,
		 MatchEnv refinementMatchEnv,
		 List filterEnvs) {
	// If filterEnvs == null, then there's no way to satisfy
	// all the conds using the current world-state.
	this.slip = slip;
	this.item = item;
	this.refinement = r;
	this.refinementMatchEnv = refinementMatchEnv;
	this.filterEnvs = filterEnvs;
    }

    boolean isExpandLater() {
	return filterEnvs == null;
    }

    boolean isStillValid() {
	return slip.MM().getVariableManager()
	         .consistentBindings(getInstantiationEnv());
    }

    void doExpansion() {
	IPlanModelManager modelManager = slip.MM();
	Debug.noteln("Expanding using", refinement);
	// Take the most complete env
	MatchEnv expandEnv = getInstantiationEnv();
	Debug.noteln("Using env", expandEnv);
	expandEnv = bindAnyOutputs(expandEnv);
	// Instantiate the refinement, create subnodes, and
	// install orderings.  item.getRefinement() is then
	// an instantiated copy.
	Map nameToChild = item.expandOneLevel(refinement, expandEnv);
	// Make any Variable bindings specified by expandEnv.
	modelManager.applyEnv(expandEnv);
	// Add the item's children to this agenda.
	Agenda actAgenda = slip.getController().getActivityAgenda();
	for (Iterator i = Collect.iterator(item.getChildren());
	     i.hasNext();) {
	    AgendaItem child = (AgendaItem)i.next();
	    actAgenda.addItem(child);
	}
	// Add issues from the refinement to the issue agenda.
	List issues = item.getRefinement().getIssues();
	for (Iterator i = Collect.iterator(issues); i.hasNext();) {
	    Issue issue = (Issue)i.next();
	    slip.getController().addIssue(issue);
	}
	// Add the orderings as time constraints
	// /\/: We will eventually have to add time constraints
	// as orderings as well, though the Refinement getOrderings()
	// might take care of that.
	// /\/: Refinement getConstraints() ought to already have
	// the orderings as time constraints?  Or not - because
	// we don't want them explicitly in the plan.
	ListOfOrdering ords = item.getRefinement().getOrderings();
	modelManager
	    .addOrderingsAsTimeConstraints(item, nameToChild, ords);
	// Add constraints from the refinement.
	// /\/: At present, they are assumed to be at begin or end
	// of self (depending on type + relation), hence apply to
	// the activity being expanded.
	List constraints = item.getRefinement().getConstraints();
	if (constraints != null) {
	    try {
		modelManager.addConstraints(item, constraints);
	    }
	    catch (FailureException fail) {
		throw slip.poison(fail);
	    }
	}
    }

    MatchEnv getInstantiationEnv() {
	// Returns an env that must be used.  If there's only
	// one branch in the filterEnvs, we can use it; otherwise,
	// we have to go back to the env from matching the
	// refinement's pattern against the activity's.
	// /\/: Not clear we should be using the filterEnvs
	// here, but as long as we also satisfy the conds
	// while expanding when there's only one filterEnv,
	// we should be ok.
	if (filterEnvs != null && !filterEnvs.isEmpty()
              && filterEnvs.size() == 1)
	    return (MatchEnv)filterEnvs.get(0);
	else
	    return refinementMatchEnv;
    }

    final Symbol S_OUTPUT_OBJECTS = Symbol.intern("output-objects");

    MatchEnv bindAnyOutputs(MatchEnv expandEnv) {
	// The outputs are a list of (var typeName) lists,
	// not necessarily LLists. /\/
	List outputs = (List)refinement.getAnnotation(S_OUTPUT_OBJECTS);
	if (outputs == null)
	    return expandEnv;
	Debug.noteln("Output Objects", outputs);
	MatchEnv env = new MatchEnv(expandEnv);
	for (Iterator i = outputs.iterator(); i.hasNext();) {
	    List pair = (List)i.next();
	    ItemVar var = (ItemVar)pair.get(0);
	    Symbol typeName = (Symbol)pair.get(1);
	    Object val = env.get(var);
	    if (val == null) {
		// The var is not bound, so we bind it to a generated
		// object name, so that the var will be replaced by
		// the object name when the refinement is instantiated.
		Symbol objName = slip.generateObjectName(typeName);
		env.put(var, objName);
	    }
	    else if (val instanceof Variable) {
		// The var is bound to a Variable.  We want to bind
		// both the var and the Variable to the same generated
		// object name.  This is typical when we're expanding
		// a node in order to achieve a cond on another node.
		// In effect, we want to pretend that the var was
		// replaced by the generated object name before
		// the var was bound to the Variable.
		Variable v = (Variable)val;
		// Debug.expect(!v.isBound());
		if (v.isBound())
		    throw slip.poison
			(var + " bound to " + v +
			 " can't be a generated " + typeName);
		Symbol objName = slip.generateObjectName(typeName);
		env.put(var, objName);
		env.put(v, objName);
	    }
	    else {
		// The var already has a value.
		throw slip.poison // new IllegalStateException
		    (var + " bound to " + val +
		     " can't be a generated " + typeName);
	    }
	}
	Debug.noteln("Env with generated objects", env);
	return env;
    }

}


