/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Mar 14 22:40:26 2005 by Jeff Dalton
 * Copyright: (c) 2004, 2005, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.io.*;
import java.util.*;

import ix.icore.plan.Plan;
import ix.iface.util.Reporting;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.xml.*;

import ix.test.OPlanPSGraph;

/**
 * A base class for GUI-less planners.  Chiefly contains utility methods.
 */
public abstract class PlannerBase extends IPlan implements Planner {

    // /\/: Need an intuitive explanation of why this extends
    // IPlan rather than the other way around.  Or, better,
    // rearrange the classes.  Multiple-inheritance would be nice.
    // Instead, maybe separate the user interface more -- then that
    // and the "algorithmic" parts could have separate inheritance
    // trees.

    // /\/: Note that here startup() doesn't do any GUI setup,
    // while IPlan, which wants to be a panel like I-P2,
    // uses the inherited Ip2 startup();

    private boolean standAlone;

    public PlannerBase(boolean standAlone) {
	super(standAlone);	// true means make this "the" agent
	this.standAlone = standAlone;
    }

    public boolean isStandAlone() {
	return standAlone;
    }

    public void replan() {	// so subclasses don't have to /\/
	throw new UnsupportedOperationException
	    ("This planning algorithm cannot replan.");
    }

    public PlanStats getStatistics() { // so subclasses don't have to /\/
	throw new UnsupportedOperationException
	    ("This planning algorithm does not produce statistics.");
    }

    /**
     * Completes setup and initialization.
     */
    public void startup() {

	// Make controller and model-manager
	modelManager = makeModelManager();
	controller = makeController();
	controller.connectTo(modelManager);

  	addHandlers();

	// Omit GUI.

	// If we're being used inside another agent.
	if (!isStandAlone())
	    return;

	readDomain(getDomain());

	// Load a plan if desired
	{
	    // We have to do this at the end, because viewers may not be
	    // able to deal with any pre-existing information.
	    String fileName = Parameters.getParameter("plan");
	    if (fileName != null)
		loadPlan(fileName);
	}

    }

    protected void startServer(Object agentName, String strategyName) {
	if (isStandAlone())
	    super.startServer(agentName, strategyName);
	// Otherwise do nothing.
    }

    /*
     * /\/: Block some inheritance from IPlan.
     */

    protected void completeStartup() {
	throw new UnsupportedOperationException("completeStartup");
    }

    public void handleInput(IPC.InputMessage message) {
	Object contents = message.getContents();
	throw new UnsupportedOperationException
	    ("Received " + Reporting.description(contents));
    }

    protected void addHandlers() {
	// No handlers
    }

    protected void initOptions() {
	// No options
	throw new UnsupportedOperationException("initOptions");
    }

    /*
     * Utilities.
     */

    protected void outputPlan() {
	String where = Parameters.getParameter("output");
	if (where == null)
	    return; // was savePlanAs();
	else {
	    Plan plan = modelManager.getPlan();
	    XML.writeObject(plan, where);
	    if (Parameters.haveParameter("output-psgraph-data"))
		outputPlanPSGraphData(plan);
	}
    }

    protected void outputPlanPSGraphData(Plan plan) {
	String where = Parameters.getParameter("output-psgraph-data");
	OPlanPSGraph psgraph = new OPlanPSGraph();
	psgraph.outputPlanForOPlan(plan, where);
    }

    protected void outputOwlSPlan() throws IOException {
	String where = Parameters.getParameter("owls-out");
	if (where == null)
	    return;
	Object writer = makeInstance("ix.util.owls.PlanToOWLS");
	if (writer == null)
	    return;
	throw new UnsupportedOperationException("OWL-S output");
    }

    protected Object makeInstance(String className) {
	// The required class may not be available, and we want to
	// avoid needing it for compilation.  /\/
	try {
	    Class c = Class.forName(className);
	    return Util.makeInstance(c);
	}
	catch (ClassNotFoundException e) {
	    Debug.noteln("Can't make class", className);
	    return null;
	}
    }


}
