/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Mar 31 19:44:53 2005 by Jeff Dalton
 * Copyright: (c) 2001 - 2005, AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.ip2.*;
import ix.ip2.event.*;
import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;
import ix.icore.process.ProcessModelManager;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

/**
 * Adds expansion actions to activities based on refinements in
 * the agent's domain.
 */
public class IPlanExpandHandler extends ExpandHandler {

    protected IPlan iplan;
    // protected Agenda actAgenda;
    // protected IPlanModelManager modelManager;

    public IPlanExpandHandler(IPlan iplan) {
	super(iplan);
	this.iplan = iplan;
	this.actAgenda = iplan.getController().getActivityAgenda();
	this.modelManager = (IPlanModelManager)iplan.getModelManager();
	iplan.getDomain().addDomainListener(this); // clean up /\/
    }

    // /\/: appliesTo is inherited.

    public void addHandlerActions(AgendaItem item) {
	item.addItemListener(new ItemListener()); // /\/
	// Expansion and manual primitive options from refinements
	// in the domain.
	for (Iterator i = iplan.getDomain().getRefinements().iterator();
	     i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    MatchEnv e = matchRefinement(item, r);
	    if (e == null) {
		continue;
	    }
//  	    else if (Collect.isEmpty(r.getNodes())
//  		     && Collect.isEmpty(r.getConstraints())) {
//  		// When there are no subnodes or constraints, the
//  		// refinement name still specifies a way of doing
//  		// it manually.
//  		item.addAction(new HandlerAction.Manual(r.getName()));
//  	    }
	    else {
		// Add action to expand using the refinement.
		item.addAction(new ExpandAction(item, r, e));
	    }
	}
    }

    // /\/: ItemListener's inherited.

    public void refinementAdded(RefinementEvent event) {
	Refinement r = event.getRefinement();
	Debug.noteln("Expand Handler noting new ", r);
	// Arrange for calls to reviseHandlerActions as needed.
	iplan.getController().reconsiderHandler(this, r);
    }

    public void reviseHandlerActions(AgendaItem item, Object reason) {
	// See if the activity (item) now has a new way to expand.
	Refinement r = (Refinement)reason;
	MatchEnv env = matchRefinement(item, r);
	if (env == null) {
	    return;
	}
//  	else if (Collect.isEmpty(r.getNodes())
//  		 && Collect.isEmpty(r.getConstraints())) {
//  	    Debug.noteln("New manual primitive for", item);
//  	    item.insertAction(new HandlerAction.Manual(r.getName()));
//  	}
	else {
	    Debug.noteln("New expansion for", item);
	    item.insertAction(new ExpandAction(item, r, env));
	}
    }

    public HandlerAction makeManualExpandAction(AgendaItem item,
						Refinement r) {
	return new ManualExpandAction(item, r);
    }

    /**
     * Handles an activity by expanding it into subactivities
     * as specified by a refinement schema.
     */
    private class ExpandAction extends HandlerAction {

	AgendaItem item;
	Refinement refinement;
	MatchEnv refinementMatchEnv;

	ListOfConstraint filters;
	List filterEnvs = Collections.EMPTY_LIST;

	ExpandAction(AgendaItem item, Refinement refinement, MatchEnv env) {
	    this.item = item;
	    this.refinement = refinement;
	    this.refinementMatchEnv = env;
	    this.filters = refinement.getFilterConstraints();
	    this.shortDescription = "Expand using " + refinement.getName();
	}

	public boolean isStillValid() {
	    if (item.isExpanded())
		return false;
	    refinementMatchEnv = matchRefinement(item, refinement);
	    return refinementMatchEnv != null;
	}

	public boolean canAlwaysBeTakenNow() {
	    return true;
	}

	public boolean isReady() {
	    // return filters.isEmpty() || !filterEnvs.isEmpty();
	    return !item.isExpanded();  // ?? /\/
	}

	public void computeStatus() {
//  	    if (!filters.isEmpty())
//  		filterEnvs =
//  		    modelManager.evalFilters(filters, refinementMatchEnv);
	}

	public void handle(AgendaItem item) {
	    Debug.expect(isReady(),
	        this + " asked to handle " + item + " when not ready");

	    // That an option's been selecte implies that execution
	    // could begin.  /\/: But subnodes must be put in first
	    // or they will get the wrong status as they are added
	    // (because parent will be executing and orderings won't
	    // yet be in so it will look like all preconditions have
	    // been met).
	    Debug.expect(item == this.item);
	    do_expansion(item);
	    item.fireAgendaItemEdited(); // /\/
	}

	protected void do_expansion(AgendaItem item) {
	    Debug.noteln("Expanding using", refinement);
	    // Take the most complete env
	    MatchEnv expandEnv = getInstantiationEnv();
	    Debug.noteln("Using env", expandEnv);
	    Debug.noteln("Filters", filters);
	    // Instantiate the refinement, create subnodes, and
	    // install orderings.  item.getRefinement() is then
	    // an instantiated copy.
	    item.expandOneLevel(refinement, expandEnv);
	    // Make any Variable bindings specified by expandEnv.
	    applyEnv(expandEnv);
	    // Add the item's children to this agenda.
	    for (Iterator i = Collect.iterator(item.getChildren());
		 i.hasNext();) {
		AgendaItem child = (AgendaItem)i.next();
		actAgenda.addItem(child);
	    }
	    // Add issues from the refinement to the issue agenda.
	    List issues = item.getRefinement().getIssues();
	    for (Iterator i = Collect.iterator(issues); i.hasNext();) {
		Issue issue = (Issue)i.next();
		iplan.getController().addIssue(issue);
	    }
	    // Add constraints from the refinement.
	    // /\/: At present, they are assumed to be at begin or end
	    // of self (depending on type + relation), hence apply to
	    // the activity being expanded.
	    List constraints = item.getRefinement().getConstraints();
	    if (constraints != null) {
		iplan.getModelManager().addConstraints(item, constraints);
	    }
	    /* - - - deleted - - -
	    // Get filter conds from the instantiated refinement.
	    List conds = item.getRefinement().getFilterConditions();
	    Debug.noteln("Instantiated conds", conds);
	    // If no conds, we're done.
	    if (conds.isEmpty()) {
		adjustStatus(item);
		return;
	    }
	    // Make sure we respect the ways the filter conds
	    // might bind now that the refinement has been
	    // instantiated.
	    // Don't need to pass anything in the env this time.
	    // The model manager returns a list of envs
	    List envs = modelManager.evalFilters(conds, new MatchEnv());
	    Debug.expect(!envs.isEmpty(),
			 "Cannot match filter conditions " + conds);
	    // The envs tell us all the ways we could satisfy
	    // all the filters by binding.  Each env is a way to
	    // satisy all the filters.
	    if (envs.size() == 1) {
		// Only one way to satisfy the conds, so we can just
		// do it.
		modelManager.satisfyConds(item, conds, (MatchEnv)envs.get(0));
		adjustStatus(item);
	    }
	    else {
		item.setHandledBy(null); // /\/
		ItemHandler condHandler = 
		    iplan.getController()
		        .findHandler
		            (Fn.isInstanceOf(IPlanConditionHandler.class));
		iplan.getController().reconsiderHandler(condHandler, this);
	    }
	    - - - deleted - - - */
	}

	protected void adjustStatus(AgendaItem item) {
	    if (item.getStatus() == Status.POSSIBLE)
		item.setStatus(Collect.isEmpty(item.getChildren())
			       ? Status.COMPLETE : Status.EXECUTING);
	}

	protected MatchEnv getInstantiationEnv() {
	    // Returns an env that must be used.  If there's only
	    // one branch in the filterEnvs, we can use it; otherwise,
	    // we have to go back to the env from matching the
	    // refinement's pattern against the activity's.
	    // /\/: Not clear we should be using the filterEnvs
	    // here, but as long as we also satisfy the conds
	    // while expanding when there's only one filterEnv,
	    // we should be ok.
	    if (!filters.isEmpty() && filterEnvs.size() == 1)
		return (MatchEnv)filterEnvs.get(0);
	    else
		return refinementMatchEnv;
	}

	protected void applyEnv(MatchEnv env) {
	    Map bindings = null; 	// create if needed
	    for (Iterator i = env.entrySet().iterator(); i.hasNext();) {
		Map.Entry e = (Map.Entry)i.next();
		Object var = e.getKey();
		if (var instanceof Variable) {
		    if (bindings == null)
			bindings = new HashMap();
		    bindings.put(var, e.getValue());
		}
	    }
	    if (bindings != null)
		modelManager.bindVariables(bindings);
	}

    }

    /**
     * Handles an item by expanding it into subitems as specified
     * explicitly by the user.
     */
    private class ManualExpandAction extends ExpandAction {

	ManualExpandAction(AgendaItem item, Refinement refinement) {
	    super(item, refinement, new MatchEnv());
	    this.shortDescription = "Expand as below";
	}

	public void handle(AgendaItem item) {
	    do_expansion(item);
	}

    }

}
