/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Apr 28 19:53:53 2005 by Jeff Dalton
 * Copyright: (c) 2004 - 2005 AIAI, University of Edinburgh
 */

package ix.iplan;

import java.util.*;

import ix.icore.*;
import ix.icore.domain.*;
import ix.icore.domain.event.*;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;
import ix.util.xml.*;

/**
 * Analyses a Domain.
 *
 * <p>This class is based on parts of O-Plan.</p>
 */
public class DomainAnalyser {

    protected Domain domain;

    protected boolean domainHasChanged = true; // true to ensure 1st analysis

    // Refinement -> Set of Refinement
    protected DirectedGraph refinementToDirectSuccessors;

    // Refinement -> Set of Refinement
    // This is the transitive closure of refinementToDirectSuccessors.
    protected DirectedGraph refinementToReachableRefinements;

    // Refinement -> Set of PatternAssignment
    protected DirectedGraph possibleConditionsTable;

    // Refinement -> Set of PatternAssignment
    protected DirectedGraph possibleEffectsTable;

    // /\/: It might be better to put the conditions and effects
    // in the "possible" tables into a canonical form, with all
    // variables replaced by ??.  That would avoid having entries
    // that are the same except for having different variable names.

    /**
     * Make an analyser for a domain.
     */
    public DomainAnalyser(Domain domain) {
	Debug.expect(domain != null, "null domain");
	this.domain = domain;
	domain.checkConsistency();
	domain.addDomainListener(new DomainListener() {
	    public void refinementAdded(RefinementEvent e) {
		domainHasChanged = true;
	    }
	});
    }

    public void analyseIfNeeded() {
	if (domainHasChanged) {
	    reset();
	    analyse();
	}
    }

    protected void reset() {
	refinementToDirectSuccessors = null;
	refinementToReachableRefinements = null;
	possibleConditionsTable = null;
	possibleEffectsTable = null;
	possibleConditionsCache.clear();
	possibleEffectsCache.clear();
	domainHasChanged = true;
    }

    /**
     * Construct tables of derived information.
     */
    public void analyse() {
	Debug.noteln("Analyzing", domain);
	// From refinement to refinement.
	refinementToDirectSuccessors = buildRefinementSuccessorTable();
	refinementToReachableRefinements =
	    Graphs.transitiveClosure(refinementToDirectSuccessors);
	DirectedGraph reflexiveReachable =
	    Graphs.makeReflexive(refinementToReachableRefinements);
	// From refinement to conditions
	FanComposer cc = new FanComposer(domain.getRefinements());
	cc.apply(reflexiveReachable);
	cc.apply(Fn.accessor(Refinement.class, "getConditions"));
	possibleConditionsTable =
	    Graphs.makeDirectedGraph(cc.getResultMap());
	// From refinement to effects
	FanComposer ec = new FanComposer(domain.getRefinements());
	ec.apply(reflexiveReachable);
	ec.apply(Fn.accessor(Refinement.class, "getEffects"));
	possibleEffectsTable =
	    Graphs.makeDirectedGraph(ec.getResultMap());
	// That's it until the domain changes again.
	domainHasChanged = false;
    }

    /* * * Queries about possible conditions * * */

    protected Map possibleConditionsCache = new WeakHashMap();  

    public Set getPossibleConditions(LList pattern) {
	Set result = (Set)possibleConditionsCache.get(pattern);
	if (result == null) {
	    result = computePossibleConditions(pattern);
	    possibleConditionsCache.put(pattern, result);
	}
	return result;
    }

    protected Set computePossibleConditions(LList pattern) {
	// Map the pattern to the refinements that can expand it
	// and then to their possible Conditions.
	FanComposer fc = new FanComposer(Lisp.list(pattern));
	fc.apply(Fn.always(refinementsExpandingPattern(pattern)));
	fc.apply(possibleConditionsTable);
	Map toConditions = fc.getResultMap(); // should have only 1 key
	Debug.expect(toConditions.size() == 1);
	return (Set)toConditions.get(pattern);
    }

    public Set getPossibleConditions(PatternAssignment pv) {
	Set result = (Set)possibleConditionsCache.get(pv);
	if (result == null) {
	    result = computePossibleConditions(pv);
	    possibleConditionsCache.put(pv, result);
	}
	return result;
    }

    protected Set computePossibleConditions(PatternAssignment pv) {
	// Map the assignment to the refinements that can satisfy it
	// and then to their possible Conditions.
	FanComposer fc = new FanComposer(Lisp.list(pv));
	fc.apply(Fn.always(refinementsForCondition(pv)));
	fc.apply(possibleConditionsTable);
	Map toConditions = fc.getResultMap(); // should have only 1 key
	Debug.expect(toConditions.size() == 1);
	return (Set)toConditions.get(pv);
    }

    /* * * Queries about possible effects * * */

    protected Map possibleEffectsCache = new WeakHashMap();  

    public Set getPossibleEffects(LList pattern) {
	Set result = (Set)possibleEffectsCache.get(pattern);
	if (result == null) {
	    result = computePossibleEffects(pattern);
	    possibleEffectsCache.put(pattern, result);
	}
	return result;
    }

    protected Set computePossibleEffects(LList pattern) {
	// Map the pattern to the refinements that can expand it
	// and then to their possible effects.
	FanComposer fc = new FanComposer(Lisp.list(pattern));
	fc.apply(Fn.always(refinementsExpandingPattern(pattern)));
	fc.apply(possibleEffectsTable);
	Map toEffects = fc.getResultMap(); // should have only 1 key
	Debug.expect(toEffects.size() == 1);
	return (Set)toEffects.get(pattern);
    }

    public Set getPossibleEffects(PatternAssignment pv) {
	Set result = (Set)possibleEffectsCache.get(pv);
	if (result == null) {
	    result = computePossibleEffects(pv);
	    possibleEffectsCache.put(pv, result);
	}
	return result;
    }

    protected Set computePossibleEffects(PatternAssignment pv) {
	// Map the assignment to the refinements that can satisfy it
	// and then to their possible effects.
	FanComposer fc = new FanComposer(Lisp.list(pv));
	fc.apply(Fn.always(refinementsForCondition(pv)));
	fc.apply(possibleEffectsTable);
	Map toEffects = fc.getResultMap(); // should have only 1 key
	Debug.expect(toEffects.size() == 1);
	return (Set)toEffects.get(pv);
    }

    /* * * Nodes that can't be expanded * * */

    public DirectedGraph getUnexpandableNodes() {
	Map map = new HashMap();
    refinementLoop:
	for (Iterator ri = domain.getRefinements().iterator(); ri.hasNext();) {
	    Refinement r = (Refinement)ri.next();
	    ListOfNodeSpec nodes = r.getNodes();
	    if (nodes == null)
		continue refinementLoop;
	    ListOfNodeSpec unexpandable = new LinkedListOfNodeSpec();
	    for (NodeSpecIterator ni = nodes.elements(); ni.hasNext();) {
		NodeSpec n = ni.next();
		if (refinementsExpandingPattern(n.getPattern()).isEmpty())
		    unexpandable.add(n);
	    }
	    if (!unexpandable.isEmpty())
		map.put(r, unexpandable);
	}
	return Graphs.makeDirectedGraph(map);
    }

    /* * * Refinement successors * * */

    protected DirectedGraph buildRefinementSuccessorTable() {
	Map map = new HashMap();
	for (Iterator i = domain.getRefinements().iterator(); i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    map.put(r, directSuccessors(r));
	}
	return Graphs.makeDirectedGraph(map);
    }

    protected Set directSuccessors(Refinement r) {
	Set result = new HashSet();
	// From expansion
	ListOfNodeSpec nodes = r.getNodes();
	if (nodes != null) {
	    for (NodeSpecIterator i = nodes.elements(); i.hasNext();) {
		NodeSpec n = i.next();
		LList pattern = n.getPattern();
		result.addAll(refinementsExpandingPattern(pattern));
	    }
	}
	// Via achieve
	List conds = r.getConditions();
	for (Iterator i = conds.iterator(); i.hasNext();) {
	    PatternAssignment pv = (PatternAssignment)i.next();
	    result.addAll(refinementsForCondition(pv));
	}
	return result;
    }

    protected Set refinementsExpandingPattern(LList pattern) {
	// /\/: A refinement's expands pattern might contain &rest,
	// but a node-spec's pattern (our parameter) won't, so the
	// refinement's pattern is used as the "pattern" when calling match.
	Set result = new HashSet();
	for (Iterator i = domain.getRefinements().iterator(); i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    if (match(r.getPattern(), pattern))
		result.add(r);
	}
	return result;
    }

    protected Set refinementsForCondition(PatternAssignment pv) {
	Set result = new HashSet();
	if (!Slip.isAchievableCond(pv.getPattern(), domain))
	    return result;	// will be empty
	for (Iterator i = domain.getRefinements().iterator(); i.hasNext();) {
	    Refinement r = (Refinement)i.next();
	    for (Iterator j = r.getEffects().iterator(); j.hasNext();) {
		PatternAssignment eff = (PatternAssignment)j.next();
		if (Slip.canBeUsedForEffect(eff.getPattern(), r)
		      && match(eff.getPattern(), pv.getPattern())
		      && match(eff.getValue(), pv.getValue()))
		    result.add(r);
	    }
	}
	return result;
    }

    /* * * Utilities * * */

    public boolean haveCommonPatterns(Collection pvs1, Collection pvs2) {
	for (Iterator i = pvs1.iterator(); i.hasNext();) {
	    PatternAssignment pv1  = (PatternAssignment)i.next();
	    for (Iterator j = pvs2.iterator(); j.hasNext();) {
		PatternAssignment pv2 = (PatternAssignment)j.next();
		if (match(pv1.getPattern(), pv2.getPattern()))
		    return true;
	    }
	}
	return false;
    }

    public boolean mightSatisfy(PatternAssignment cond,
				PatternAssignment effect) {
	return match(cond.getPattern(), effect.getPattern())
	    && match(cond.getValue(), effect.getValue());
    }

    public boolean mightSatisfy(PatternAssignment cond,
				Collection effects) {
	for (Iterator i = effects.iterator(); i.hasNext();) {
	    PatternAssignment e = (PatternAssignment)i.next();
	    if (mightSatisfy(cond, e))
		return true;
	}
	return false;
    }

    /**
     * Simple pattern-matcher that treats all variables as wildcards.
     */
    private boolean match(Object pat, Object dat) {
	if (pat == dat)
	    return true;
	else if (pat instanceof ItemVar || pat instanceof Variable)
	    return true;
	else if (dat instanceof ItemVar || dat instanceof Variable)
	    return true;
	else if (pat instanceof Cons) {
	    if (dat instanceof Cons) {
		LList p = (LList)pat;
		LList d = (LList)dat;
		while (p != Lisp.NIL && d != Lisp.NIL) {
		    Object a = p.car();
		    if (a == Matcher.REST) return true;
		    if (!match(a, d.car())) return false;
		    p = p.cdr();
		    d = d.cdr();
		}
		return p == d;	// same length?
	    }
	    else
		return false;
	}
	else
	    return pat.equals(dat);
    }

    /* * * Standalone application * * */

    /**
     * Test program.
     */
    public static void main(String[] argv) {
	Debug.off();
	Parameters.processCommandLineArguments(argv);
	String domainName = Parameters.getParameter("domain");
	Domain domain = (Domain)XML.readObject(Domain.class, domainName);
	DomainAnalyser ann = new DomainAnalyser(domain);
	ann.analyse();
	printGraph("Successor table:", ann.refinementToDirectSuccessors);
	printGraph("Reach table:", ann.refinementToReachableRefinements);
	printGraph("Possible effects:", ann.possibleEffectsTable);
	printCollection("Unreachable refinements:",
			Graphs.minimalElements
			  (ann.refinementToDirectSuccessors));
	printGraph("Unexpandable nodes:", ann.getUnexpandableNodes());
	if (Parameters.getBoolean("ask-for-conditions")) {
	    ann.askForConditionsLoop();
	}
	if (Parameters.getBoolean("ask-for-effects")) {
	    ann.askForEffectsLoop();
	}
    }

    void askForConditionsLoop() {
	askLoop("conditions", new Function1() {
	    public Object funcall(Object a) {
		return a instanceof LList
		    ? getPossibleConditions((LList)a)
		    : getPossibleConditions((PatternAssignment)a);
	    }
	});
    }

    void askForEffectsLoop() {
	askLoop("effects", new Function1() {
	    public Object funcall(Object a) {
		return a instanceof LList
		    ? getPossibleEffects((LList)a)
		    : getPossibleEffects((PatternAssignment)a);
	    }
	});
    }

    void askLoop(String resultDescr, Function1 f) {
	println("Looking for", resultDescr);
	println("Input can be a pattern or pattern = value.");
	println("A pattern should not have parens around it.");
	for (;;) {
	    try {
		String line = Util.askLine("Input:");
		if (line.equals("bye"))
		    return;
		Object query;
		if (line.indexOf("=") >= 0) {
		    String[] parts = Strings.breakAtFirst("=", line);
		    query = new PatternAssignment
			     (Lisp.elementsFromString(parts[0]),
			      Lisp.readFromString(parts[1]));
		}
		else {
		    query = Lisp.elementsFromString(line);
		}
		println("Query:", query);
		printCollection(resultDescr, (Collection)f.funcall(query));
	    }
	    catch (Throwable t) {
		Debug.noteException(t);
	    }
	}
    }

    static void printGraph(String label, DirectedGraph g) {
	// /\/: Assumes the vertexes are refinements.
	println(label);
	List refinements = sortCollection(g.getAllNodes());
	for (Iterator i = refinements.iterator(); i.hasNext();) {
	    Object v = i.next();
	    println("   ", v);
	    List successors = sortCollection(g.getSuccessors(v));
	    for (Iterator j = successors.iterator(); j.hasNext();) {
		println("      ", j.next());
	    }
	    println("");
	}
    }

    static void printCollection(String label, Collection c) {
	println(label);
	for (Iterator i = sortCollection(c).iterator(); i.hasNext();) {
	    println("   ", i.next());
	}
	println("");
    }

    static List sortCollection(Collection c) {
	List lis = new ArrayList(c);
	if (!lis.isEmpty() && lis.get(0) instanceof Refinement)
	    Collections.sort(lis, new Refinement.NameComparator());
	return lis;
    }

    private static void print(String s) {
	System.out.print(s);
    }

    private static void println(String s) {
	System.out.println(s);
    }

    private static void println(String s, Object o) {
	System.out.println(s + " " + o);
    }

}
