/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Aug 23 17:49:35 2005 by Jeff Dalton
 * Copyright: (c) 2003 - 2005, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.Variable;
import ix.icore.domain.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.*;

public class VariableManager extends MatchChoiceManager {

    protected final boolean MATCH_DEBUG = false;

    protected boolean noticeForced = true;

    protected Ip2ModelManager modelManager;

    protected ComputeInterpreter computeInterpreter;

    public VariableManager(Ip2ModelManager modelManager) {
	this.modelManager = modelManager;
	this.computeInterpreter = modelManager.makeComputeInterpreter();
    }

    ComputeInterpreter getComputeInterpreter() {
	return computeInterpreter;
    }

    public void showState() {
	if (MATCH_DEBUG) super.showState();
    }

    /**
     * Returns a list of MatchEnvs, one for each way in which all
     * of the conditions can be satisfied together.
     */
    public List evalFilters(ListOfConstraint conds, Map state, MatchEnv env) {
	return new FilterMatcher(state).evalFilters(conds, env);
    }

    /**
     * Returns a list of the conditions that can be satisfied
     * individually.  This is normally called when they cannot
     * all be satisfied together, because the result may help
     * determine what the problem is.
     */
    public ListOfConstraint testFilters(ListOfConstraint conds,
					Map state,
					MatchEnv env) {
	return new FilterMatcher(state).testFilters(conds, env);
    }

    public void tryBindings(Map bindings) {
	Debug.noteln("Var manager trying", bindings);
	try {
	    noticeForced = false;
	    MatchChoice choice =
		new MatchChoice(Lisp.list(new Bindings(bindings)));
	    add(choice);
	    try {
		recalculate();
		if (choice.getLiveBranches().isEmpty()) {
		    Debug.noteln("tryBindings fails for", bindings);
		    super.showState(); // ignores MATCH_DEBUG
		    throw new IllegalArgumentException
			("Cannot apply binding combination " +
			 bindings);
		}
	    }
	    finally {
		// /\/: Any exception thrown in a finally will
		// cause an exception from the try to be lost.
		remove(choice);
		recalculate();
	    }
	}
	finally {
	    noticeForced = true;
	}
    }

    public void newBindings(Map bindings) {
	// N.B. The bindings have already been applied.
	Debug.noteln("Var manager sees bindings", bindings);
	// Add the bindings as a 1-branch MatchChoice.
	// /\/: This is kind of a hack.
	add(new MatchChoice(Lisp.list(new Bindings(bindings))));
	recalculate();
    }

    public List getConstraints() {
	List constraints = new LinkedList();
	for (Iterator i = matchChoices.iterator(); i.hasNext();) {
	    MatchChoice mc = (MatchChoice)i.next();
	    if (!Variable.unboundVarsIn(mc.getVariables()).isEmpty())
		// If the vars in the match-choice don't all already
		// have values, include it as a constraint.
		constraints.add(mc.toConstraint());
	}
	return constraints;
    }

    /**
     * Called within the variable manager when it notices that some
     * variables have only one possible value remaining.
     */
    protected MatchEnv noticeForcedBindings() {
	// /\/: Super method returns a value only for use here.
	if (noticeForced == false)
	    return null;
	MatchEnv forced = super.noticeForcedBindings();
	if (!forced.isEmpty())
	    modelManager.forcedBindings(forced); // /\/ temporary???
	return forced;
    }

    protected class FilterMatcher {

	protected Map stateMap;
	protected List resultEnvs;

	FilterMatcher(Map stateMap) {
	    this.stateMap = stateMap;
	}

	List evalFilters(ListOfConstraint conds, MatchEnv env) {
	    // -> List<MatchEnv>
	    if (MATCH_DEBUG) Debug.noteln("\nPMM evaluating filters", conds);
	    Debug.expect(!conds.isEmpty());
	    Debug.expect(resultEnvs == null);
	    resultEnvs = new LinkedList();
	    try {
		LList cl = LList.newLList(conds);
		filter((Constraint)cl.car(), cl.cdr(), env);
		return resultEnvs;
	    }
	    finally {
		resultEnvs = null;
	    }
	}

	ListOfConstraint testFilters(ListOfConstraint conds,
				     MatchEnv baseEnv) {
	    ListOfConstraint satisfiable = new LinkedListOfConstraint();
	condLoop:
	    for (Iterator i = conds.iterator(); i.hasNext();) {
		Constraint c = (Constraint)i.next();
		if (c.getType() != Refinement.S_WORLD_STATE)
		    continue;	// /\/ for now
		PatternAssignment pv = c.getPatternAssignment();
	    stateLoop:
		for (Iterator m = stateMap.entrySet().iterator()
			 ; m.hasNext();) {
		    Map.Entry entry = (Map.Entry)m.next();
		    MatchEnv env = matchFilter(pv, entry, baseEnv);
		    if (env != null) {
			satisfiable.add(c);
			continue condLoop;
		    }
		}
	    }
	    return satisfiable;
	}

	protected void filter(Constraint c,
			      LList restConds,
			      MatchEnv baseEnv) {
	    if (c.getType() == Refinement.S_COMPUTE)
		evalComputeFilter(c, restConds, baseEnv);
	    PatternAssignment pv = c.getPatternAssignment();
	    if (MATCH_DEBUG)
		Debug.noteln("PMM filter " + pv + " in env " + baseEnv);
	    // /\/: The first word might be a variable.
	    // Object word1 = pv.getPattern().car();
	    for (Iterator m = stateMap.entrySet().iterator(); m.hasNext();) {
		Map.Entry entry = (Map.Entry)m.next();
		// if (((LList)entry.getKey()).car() != word1)
		//     continue;
		MatchEnv env = matchFilter(pv, entry, baseEnv);
		if (env != null) {
		    if (restConds.isEmpty()) {
			// We've matched all conds, so remember the result.
			if (!resultEnvs.contains(env))
			    resultEnvs.add(env);
		    }
		    else {
			// We've matched one cond, now on to the next.
			filter((Constraint)restConds.car(),
			       restConds.cdr(),
			       env);
		    }
		}
		// Try the next entry to match the cond against
	    }
	}

	protected MatchEnv matchFilter(PatternAssignment pv,
				       Map.Entry entry,
				       MatchEnv baseEnv) {
	    if (MATCH_DEBUG)
		Debug.noteln("Trying " + pv + " =?= " + entry);
	    MatchEnv env = new MatchEnv(baseEnv); // copy /\/
	    env = Matcher.match(pv.getPattern(), entry.getKey(), env);
	    if (env == null)
		return null;
	    env = Matcher.match(pv.getValue(), entry.getValue(), env);
	    if (env == null)
		return null;
	    else if (consistentBindings(env)) {
		if (MATCH_DEBUG)
		    Debug.noteln("Matched " + pv + " to " + entry +
				 " giving " + env);
		return env;
	    }
	    else {
		Debug.noteln("Constraints rejected", entry);
		return null;
	    }
	}

	protected void evalComputeFilter(Constraint c, LList restConds,
					 MatchEnv baseEnv) {
	    LList pattern = c.getPattern();
	    LList form;
	    try { form = (LList)fullyInstantiate(pattern, baseEnv); }
	    catch (FailureException fail) {
		return;
	    }
	    Debug.noteln("Evaluating compute", form);
	    Object value = computeInterpreter.compute(form);
	    Debug.noteln("Value of " + form + " is " + value);
	    if (c.getRelation() == Refinement.S_MULTIPLE_ANSWER) {
		if (!(value instanceof Collection)) {
		    // /\/: Just fail here?  Or error?  At least warn for now.
		    Debug.warn("Multiple answer for " + c +
			       " is not a collection; instead it is: " +
			       value);
		    return;
		}
		for (Iterator i = ((Collection)value).iterator()
			 ; i.hasNext();) {
		    matchComputeResult(c, i.next(), restConds, baseEnv);
		}
	    }
	    else {
		matchComputeResult(c, value, restConds, baseEnv);
	    }
	}

	final Symbol TRUE = Symbol.intern("true");
	final Symbol FALSE = Symbol.intern("false");

	protected void matchComputeResult(Constraint c, Object value,
					  LList restConds, MatchEnv baseEnv) {
	    Object valuePattern = c.getValue();
	    MatchEnv env = new MatchEnv(baseEnv); // copy /\/
	    if (valuePattern == TRUE)
		env = computeInterpreter.isTrue(value) ? env : null;
	    else if (valuePattern == FALSE)
		env = !computeInterpreter.isTrue(value) ? env : null;
	    else
		env = Matcher.match(valuePattern, value, env);
	    if (env != null && consistentBindings(env)) {
		if (restConds.isEmpty()) {
		    // We've matched all conds, so remember the result.
		    if (!resultEnvs.contains(env))
			resultEnvs.add(env);
		}
		else {
		    // We've matched one cond, now on to the next.
		    filter((Constraint)restConds.car(),
			   restConds.cdr(),
			   env);
		}
	    }
	}

    }

    public boolean consistentBindings(MatchEnv env) {
	for (Iterator i = env.entrySet().iterator(); i.hasNext();) {
	    Map.Entry e = (Map.Entry)i.next();
	    Object k = e.getKey();
	    Object v = e.getValue();
	    if (k instanceof Variable)
		if (!consistentBinding((Variable)k, v))
		    return false;
	}
	return true;
    }

    protected boolean consistentBinding(Variable var, Object val) {
	// /\/: Inefficient
	if (!getVariables().contains(var)
	    || getPossibleValues(var).contains(val))
	    return true;
	Debug.noteln("Can't bind " + var + " to " + val +
		     " because the only possible values are " +
		     getPossibleValues(var));
	Debug.noteln("Vars are", getVariables());
	showState();
	return false;
    }

    protected Object fullyInstantiate(Object item, final MatchEnv env) {
	ObjectCopier copier = new DeepCopier() {
	    // N.B. this assumes that variable values do not contain
	    // variables.
	    public Object copy(Object obj) {
		// /\/: This isn't quite right, because if a var
		// appears inside a quoted object, we don't need to
		// put QUOTE around its value.  So for now, we'll
		// just say that case isn't allowed.
		if (obj instanceof Variable || obj instanceof ItemVar)
		    // /\/: Unfortunatey, quoting doesn't work, because
		    // some of the variables may already have been
		    // replaced by their values.
 		    // return quote(deref(obj));
		    return deref(obj);
		else
		    return super.copy(obj);
	    }
	    private Object deref(Object obj) {
		if (obj instanceof Variable) {
		    Variable var = (Variable)obj;
		    Object val = var.getValue();
		    if (val == null)
		        val = env.get(var);
		    if (val == null)
		        throw new FailureException();
		    else
		        return deref(val); // more derefs may be needed
		}
		else if (obj instanceof ItemVar) {
		    ItemVar var = (ItemVar)obj;
		    Object val = env.get(var);
		    if (val == null)
		        throw new FailureException();
		    else
		        return deref(val); // more derefs may be needed
		}
		else {
		    return obj;
		}
	    }
	    private LList quote(Object item) {
		return Lisp.list(QUOTE, item);
	    }
	    private final Symbol QUOTE = Symbol.intern("quote");
	};
	return copier.copy(item);
    }

}
