/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sat Nov 13 17:13:30 2004 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import javax.swing.*;

import ix.iface.util.ToolController;
import ix.icore.*;
import ix.icore.domain.Domain;
import ix.util.*;
import ix.util.lisp.*;

/**
 * Generates a {@link HandlerAction} that loads a new domain
 * and one that inclides a domain in the agent's existing domain.
 */
public class LoadDomainHandler extends ActivityHandler {

    public static final Symbol
	S_NEW_DOMAIN = Symbol.intern("new-domain"),
	S_INCLUDE_DOMAIN = Symbol.intern("include-domain"),
	S_LOAD_DOMAIN = Symbol.intern("load-domain");

    protected Ip2 ip2;

    public LoadDomainHandler(Ip2 ip2) {
	super("Load or include a domain");
	this.ip2 = ip2;
    }

    public List getSyntaxList() {
	return (LList)Lisp.readFromString
	    ("((new-domain ?domain)" +
	     " (include-domain ?domain)" +
	     " (load-domain ?domain))");
    }

    public boolean appliesTo(AgendaItem item) {
	LList pattern = item.getPattern();
	Object verb = getVerb(pattern);
	return pattern.length() == 2
	    && (verb == S_NEW_DOMAIN || verb == S_INCLUDE_DOMAIN ||
		verb == S_LOAD_DOMAIN);
    }
	
    public void addHandlerActions(AgendaItem item) {
	Object verb = getVerb(item.getPattern());
	if (verb == S_NEW_DOMAIN) {
	    item.addAction(new NewDomainAction(item));
	}
	else if (verb == S_INCLUDE_DOMAIN) {
	    item.addAction(new IncludeDomainAction(item));
	}
	else if (verb == S_LOAD_DOMAIN) {
	    item.addAction(new IncludeDomainAction(item));
	    item.addAction(new NewDomainAction(item));
	}
	else
	    throw new ConsistencyException("Invalid verb");
    }

    protected Object getVerb(LList pattern) {
	return pattern.length() > 0 ? pattern.get(0) : null;
    }

    public void handle(AgendaItem item) {
	throw new ConsistencyException(this + " called directly");
    }

    abstract class AbstractLoadDomainAction extends HandlerAction {

	AgendaItem item;

	AbstractLoadDomainAction(AgendaItem item) {
	    this.item = item;
	}

	public boolean isReady() {
	    return Variable.isFullyBound(item.getPattern());
	}

	public ActionUnreadyReason getUnreadyReason() {
	    return new UnboundVariablesUnreadyReason(this, item);
	}

	Domain getDomain(LList pattern) {
	    LList pat = (LList)Variable.removeVars(pattern);
	    Object spec = pattern.get(1);
	    if (spec instanceof Domain)
		return (Domain)spec;
	    else if (spec instanceof Symbol || spec instanceof String)
		return readDomain(spec.toString());
	    else
		throw new IllegalArgumentException
		    (spec + " is not a valid reference to a domain");
	}

	Domain readDomain(String resourceName) {
	    Domain result = new Domain();
	    ip2.readDomain(result, resourceName);
	    return result;
	}

	boolean existsDomainEditor() {
	    ToolController tc =
		ip2.frame.toolManager.findTool("Domain Editor");
	    return tc != null && tc.getTool() != null;
	}

	boolean askIfSafe() {
	    switch(JOptionPane.showConfirmDialog
		   (ip2.frame,
		    "Have you saved any changes made in the domain editor?",
		    "Confirm",
		    JOptionPane.YES_NO_OPTION)) {
	    case JOptionPane.YES_OPTION:
		return true;
	    case JOptionPane.NO_OPTION:
		return false;
	    default:
		throw new ConsistencyException("Not yes or no");
	    }
	}

    }

    class NewDomainAction extends AbstractLoadDomainAction {

	NewDomainAction(AgendaItem item) {
	    super(item);
	    this.shortDescription = "New domain";
	}

	public void handle(AgendaItem item) {
	    Debug.expect(item == this.item);
	    if (!existsDomainEditor() || askIfSafe()) {
		Domain domain = getDomain(item.getPattern());
		ip2.getDomain().clear();
		ip2.getDomain().takeFrom(domain);
		item.setStatus(Status.COMPLETE);
	    }
	}

    }

    class IncludeDomainAction extends AbstractLoadDomainAction {

	IncludeDomainAction(AgendaItem item) {
	    super(item);
	    this.shortDescription = "Include domain";
	}

	public void handle(AgendaItem item) {
	    Debug.expect(item == this.item);
	    if (!existsDomainEditor() || askIfSafe()) {
		Domain domain = getDomain(item.getPattern());
		ip2.getDomain().takeFrom(domain);
		item.setStatus(Status.COMPLETE);
	    }
	}

    }

}
