/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Wed Jun 21 15:25:12 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;
import java.lang.reflect.Constructor;

import ix.icore.*;
import ix.util.*;

import ix.util.reflect.ClassFinder;
import ix.util.reflect.DashSyntaxClassFinder;

/**
 * Generates HandlerActions for an AgendaItem.
 *
 * <p>N.B. For Handlers that can handle both issues and activities,
 * extend this class rather than IssueHandler or ActivityHandler.</p>
 */
public abstract class ItemHandler {

    /** A short description of the action(s) implemented by this handler. */
    protected String actionDescription;

    /**
     * Constructs a handler with the given description.
     *
     * @see #getActionDescription()
     */
    public ItemHandler(String actionDescription) {
	this.actionDescription = actionDescription;
    }

    /**
     * Returns a short description of the action(s) implemented
     * by this handler.
     */
    public String getActionDescription() {
	return actionDescription;
    }

    /**
     * Returns a list of patterns representing the item pattern
     * syntaxes this handler can process, or null if no particular
     * syntax is required.  Note that this information is <i>not</i>
     * automatically used when determining whether this handler should
     * add actions to an agenda item.  Instead, this handler's
     * contribution to that decision is supplied by the
     * {@link #appliesTo(AgendaItem)} method.
     *
     * @see AgendaItem#getPattern()
     */
    public List getSyntaxList() {
	return null;
    }

    /**
     * Gives this handler a say in whether it should add
     * actions to an item.  If this method returns false for
     * an AgendaItem, or the item's wantsActionsFrom method
     * returns false for this handler, this handler will not
     * be asked to add actions to that item.  
     *
     * @see AgendaItem#wantsActionsFrom(ItemHandler handler)
     */
    public boolean appliesTo(AgendaItem item) {
	// This method allows the handler to reject some items.
	// This is especially useful if the handler is added as
	// a "universal" handler.
	return true;
    }

    /**
     * Indicates whether this handler has the authority to
     * handle items automatically rather than only when
     * explicitly invoked.
     */
    public boolean isAutomatic() {
	return false;
    }

    /**
     * Called to add any {@link HandlerAction}s that this
     * handler regards as appropriate for the specified AgendaItem.
     *
     * <p>This method is called only if both this handler's
     * {@link #appliesTo(AgendaItem)} and the specified item's
     * {@link AgendaItem#wantsActionsFrom(ItemHandler)} methods
     * return true.</p>
     *
     * <p>The method provided by the ItemHandler class just adds
     * an action that calls this handler's {@link #handle(AgendaItem)}
     * method.  This allows suitably simple handlers to avoid
     * working directly with HandlerActions.</p>
     *
     * <p>Note that actions are added by calling an item's
     * {@link AgendaItem#addAction(HandlerAction)} method.</p>
     */
    public void addHandlerActions(AgendaItem item) {
	// It's assumed the controller has already determined that
	// this is an appropriate handler for the item.
  	item.addAction
  	    (new HandlerAction.Automatic(this));
    }

    /**
     * Called after something has happened that may require new
     * HandlerActions to be added to an item.  Typically, this
     * method is called as a consequnce of an ItemHandler --
     * usually this ItemHandler -- calling the controller's
     * reconsiderHandler method.  In other words, the usual case
     * is when an ItemHandler causes its own reviseHandlerActions
     * method to be called; and consequencetly the handler may be
     * able to agree with itself (so to speak) about what various
     * "reason" objects mean.
     *
     * <p>The reviseHandlerActions method is very similar to
     * the {@link #addHandlerActions(AgendaItem)} method in
     * terms of its function and the applicability conditions
     * on when it is called.  The main difference is that it
     * is called to look at AgendaItems that are not having
     * actions added for the first time and that may already
     * have an appropriate action, so that no new one needs
     * to be added.</p>
     *
     * <p>Note that the usual way for an ItemHandler to get
     * HandlerActions <i>removed</i> is for it to call the controller's
     * checkActionValidity method.  That results in calls to
     * {@link HandlerAction#isStillValid()} methods.</p>
     *
     * @see PanelController#reconsiderHandler(ItemHandler handler,
     *                                        Object reason)
     * @see PanelController#checkActionValidity(ItemHandler handler,
     *                                          Object reason)
     */
    public void reviseHandlerActions(AgendaItem item, Object reason) {
	throw new UnsupportedOperationException
	    (this + "cannot revise handler actions");
    }

    /**
     * Called to perform the action implemented by this handler,
     * if the selected HandlerAction so desires.
     *
     * @see #addHandlerActions(AgendaItem)
     */
    public void handle(AgendaItem item) {
	Debug.noteln(this + " handling " + item);
    }

    /**
     * Returns a textual representation of this handler.
     */
    public String toString() {
	return "ItemHandler[" + getActionDescription() + "]";
    }

    /*
     * Handler construction
     */

    private static ClassFinder handlerClassFinder =
	// /\/: We know this will also accept Java-syntax names.
	new DashSyntaxClassFinder() {
	    protected void addInitialImports() {
		addImport("ix.ip2.*");
		addImport("ix.test.*");
	    }
	};

    /**
     * Constructs an instance of the class with the specified name.
     * The class must exist and be loadable, must be a subclass of
     * ItemHandler, and must have either a zero-argument constructor
     * (which is tried first) or a one-argument constructor with
     * the argument declared to be of class Ip2.
     */
    public static ItemHandler makeHandler(Ip2 ip2, String className) {
	Class c = handlerClassFinder.classForName(className);
	if (c == null)
	    throw new IllegalArgumentException
		("Cannot find a handler class named " + 
		 Strings.quote(className));
	if (!(ItemHandler.class.isAssignableFrom(c)))
	    throw new IllegalArgumentException
		("Class " + className + " is not a handler class.");
	try {
	    Constructor cons0 = c.getConstructor(new Class[]{});
	    return (ItemHandler)cons0.newInstance(new Object[]{});
	}
	catch (Exception e1) {
	    try {
		Constructor cons1 = c.getConstructor(new Class[]{Ip2.class});
		return (ItemHandler)cons1.newInstance(new Object[]{ip2});
	    }
	    catch (Exception e2) {
		throw new IllegalArgumentException
		    ("Cannot make an instance of handler class " + className +
		     " because trying to use the () constructor" +
		     " threw " + Debug.describeException(e1) +
		     " and trying to use the (Ip2) constructor" +
		     " threw " + Debug.describeException(e2));
	    }
	}
	
    }

}
