/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Tue Aug 22 16:52:34 2006 by Jeff Dalton
 * Copyright: (c) 2002, 2004 - 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.util.*;

/**
 * One of the ways in which an agenda item might be handled.
 * HandlerActions are added to {@link AgendaItem}s by
 * {@link ItemHandler}s.
 */
public class HandlerAction {

    /**
     * Briefly describes what this action does.
     */
    protected String shortDescription;

    /**
     * Constructor used by subclasses.
     */
    public HandlerAction() { }

    /**
     * Returns a brief description of how this action would handle
     * an item.
     */
    public String getActionDescription() {
	return shortDescription;
    }

    /**
     * Handle the item in the manner appropriate to this handler-action.
     * The method in the HandlerAction class merely changes the item's
     * status to COMPLETE and is overridden in subclasses that handle
     * the item in more interesting ways.
     */
    public void handle(AgendaItem item) {
	Debug.noteln("HandlerAction " + shortDescription +
		     " handling item " + item);
	item.setStatus(Status.COMPLETE);
    }

    /**
     * Indicates whether this action can be taken now regardless
     * of the current status of the {@link AgendaItem}.
     * It is called by the item's
     * {@link AgendaItem#actionCanBeTakenNow(HandlerAction)} method
     * and, if it returns true, overrides the value that would
     * be returned by that method.
     *
     * <p>The method defined in the HandlerAction class returns false.</p>
     */
    public boolean canAlwaysBeTakenNow() {
	return false;
    }

    // /\/: Should isStillValid(), isReady(), and computeStatus()
    // also have an AgendaItem parameter, or else should the
    // handleItem method not have that parameter?

    /**
     * Called when something has happened that might mean this action
     * no longer makes sense and should be deleted.  The method in
     * the HandlerAction class just returns true and is overridden
     * in subclasses that have instances that may cease to be valid.
     *
     * <p>Typically, this method is called as a consequence of
     * an {@link ItemHandler} calling the controllers's
     * checkActionValidity method.</p>
     *
     * @see PanelController#checkActionValidity(ItemHandler handler,
     *                                          Object reason)
     */
    public boolean isStillValid() {
	return true;
    }

    /**
     * Indicates whether this action is ready to be used or else
     * is waiting for something that would enable it to be used.
     * The method in the HandlerAction class just returns true
     * and is overridden in subclasses that have instances that
     * are not always ready.
     *
     * @see #getUnreadyReason()
     */
    public boolean isReady() {
	return true;
    }

    /**
     * Called when something has happened that might have changed
     * this action's ready status.  The method in the HandlerAction
     * class does nothing and is overridden in subclasses that have
     * instances that are not always ready.
     */
    public void computeStatus() {
    }

    /**
     * Returns an object that explains why this handler is
     * not ready to be used.
     *
     * @see #isReady()
     */
    public ActionUnreadyReason getUnreadyReason() {
	return null;
    }

    public String toString() {
	return "HandlerAction[" + shortDescription + "]";
    }


    /*
     * Some simple subclasses
     */

    /**
     * Handles an item by doing nothing.  Used when it's
     * convenient to select an action without actually handling
     * the item, so that all other actions remain available
     * and can be selected later on.
     */
    public static class NoAction extends HandlerAction {

        // public static final String NO_ACTION_DESCRIPTION = "No Action";
        public static final String NO_ACTION_DESCRIPTION = "   ";

 	public NoAction() {
	    shortDescription = NO_ACTION_DESCRIPTION;
	}

	public void handle(AgendaItem item) {
	    // N.B. no status change.
	    Debug.noteln("No action for", item);
	    // Have to clear the handledBy field.
	    Debug.expect(item.getHandledBy() == this);
	    item.setHandledBy(null);
	}

    }

    /**
     * Handles an issue by invoking one of the agent-level
     * issue-handlers.
     */
    public static class Automatic extends HandlerAction {

	ItemHandler handler;

	public Automatic(ItemHandler h) {
	    this(h, /* "Do it using " + */ h.getActionDescription());
	}

	public Automatic(ItemHandler h, String descr) {
	     this.handler = h;
	     this.shortDescription = descr;
	}

	public void handle(AgendaItem item) {
	    Debug.noteln("Giving item to handler", handler);
	    handler.handle(item);
	    // N.B. It is up to that handler to do:
	    //   i.setStatus(STATUS_COMPLETE);
	    // or whatever other status changes are required.
	}

    }

    /**
     * Handles an issue by invoking one of the agent-level
     * issue-handlers, but does not become ready until all
     * of the variables in its AgendaItem's pattern are bound.
     */
    public static class AutomaticWhenBound extends Automatic {

	// /\/ The AgendaItem is needed only because isReady()
	// and getUnreadyReason() do not have it as a parameter.

	AgendaItem item;

	public AutomaticWhenBound(AgendaItem item, ItemHandler h) {
	    super(h);
	    this.item = item;
	}

	public AutomaticWhenBound(AgendaItem item, ItemHandler h,
				  String descr) {
	    super(h, descr);
	    this.item = item;
	}

	/** Returns true iff all variables in the pattern have values. */
	public boolean isReady() {
	    return Variable.isFullyBound(item.getPattern());
	}

	/** 
	 * Returns a reason giving the unbound variables
	 * in the agenda-item's pattern.
	 */
	public ActionUnreadyReason getUnreadyReason() {
	    return new UnboundVariablesUnreadyReason(this, item);
	}

    }

    /**
     * Indicates that the item has been handled by the user.
     * Note that an item may have more than one Manual action,
     * with different descriptions that act as reminders of
     * different ways the user might act.
     */
    public static class Manual extends HandlerAction {
	public Manual() {
	    shortDescription = "Done";
	}
	public Manual(String howDescription) {
	    shortDescription = "Done using " + howDescription;
	}
    }

    /**
     * Used to indicate the the item does not apply in the current
     * situation.
     */
    public static class NotApplicable extends Manual {
	public NotApplicable() {
	    shortDescription = "N/A";
	}
    }

}

// Issues:
// * The approach taken by isStillValid() and computeStatus() is
//   obviously less than ideal, because something outside the object
//   has to know when to call the method; but it will do for now.
