/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Fri Jan 16 02:56:37 2004 by Jeff Dalton
 * Copyright: (c) 2001 - 2004, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.*;
import ix.ispace.*;
import ix.ispace.event.*;

import ix.util.*;

/**
 * Generates HandlerActions to forward Issues or Activities to other agents.
 */
public class ForwardingHandler extends ItemHandler implements ContactListener {

    protected IXAgent agent;
    protected String action;
    protected AgentRelationship rel;
    protected boolean reportBack;

    protected boolean capabilityDefault = true;

    protected ContactManager contactManager;

    public ForwardingHandler(IXAgent agent,
			     String action,
			     AgentRelationship rel,
			     boolean reportBack) {
	super(action + " to " + rel + " agents"); // action description
	this.agent = agent;
	this.action = action;
	this.rel = rel;
	this.reportBack = reportBack;

	this.contactManager = agent.getContactManager();
	this.contactManager.addContactListener(this);
    }

    public void addHandlerActions(AgendaItem item) {
	// It's assumed the controller has already determined that
	// this is an appropriate handler for the item.
	List toNames = getToNames(item);
	Debug.noteln(this + " adding actions for agents " + toNames);
	for (Iterator i = toNames.iterator(); i.hasNext();) {
	    String to = (String)i.next();
	    item.addAction(makeForwardingAction(to, item));
	}
    }

    public ForwardingAction makeForwardingAction
                                (String toName, AgendaItem about) {
	return new ForwardingAction(toName, about);
    }

    protected List getToNames(AgendaItem item) {
	return (List)Collect.map(new LinkedList(), // in case empty /\/
				 getToAgents(item),
				 Fn.accessor(AgentData.class, "getName"));
    }

    public List getToAgents(AgendaItem item) {
	// Return agents that have the required relationship and
	// the capability that corresponds to the item's pattern's verb,
	// assuming that if an agent's capabilities are unknown, then
	// they might include the required one.
	Capability c = VerbCapability.from(item.getPattern());
	List relData = contactManager.getAgentData(rel);
	List capData = contactManager.getAgentData(c, capabilityDefault);
	Debug.noteln("Agents with " + rel, relData);
	Debug.noteln("Agents with " + c, capData);
	if (relData == null || capData == null)
	    return Collections.EMPTY_LIST;
	else
	    return (List)Collect.intersection(relData, capData);
    }

    public void handle(AgendaItem item) {
	throw new Error(this + " called directly");
    }

    public void contactChange(ContactEvent e) {
	// First, any action deletions.
	// This causes calls to action isStillValid metods.
	((Ip2)agent).getController().checkActionValidity(this, e );
	// Now see if any actions might need to be added.
	// N.B. At this point, we don't know what capability is
	// be required, because it varies with the AgendaItem.
	if (e.isNewContact() || e.isRelationshipChange()
	    || e.isCapabilityChange()) {
	    AgentData data = e.getNewData();
	    AgentRelationship r = data.getRelationship();
	    if (r == rel) {
		Debug.noteln("Possible new forwarding target", data);
		((Ip2)agent).getController().reconsiderHandler(this, e);
		// Causes calls to reviseHandlerActions(i, r) for various i.
	    }
	}
    }

    public void reviseHandlerActions(AgendaItem item, Object reason) {
	// called after a relevant contact change.
	ContactEvent e = (ContactEvent)reason;
	AgentData data = e.getNewData();
	Debug.expect(data.getRelationship() == rel);

	if (data.hasCapability(VerbCapability.from(item.getPattern()),
			       capabilityDefault)) {
	    String toName = data.getName();
	    HandlerAction f = makeForwardingAction(toName, item);
	    if (item.findAction(f.getActionDescription()) == null) {
		// /\/: Shouldn't have to check there isn't already
		// such an action.  Also that's not the right check -
		// an explicity class check is needed.
		item.insertAction(f);
	    }
	}
    }

    public String toString() {
	return "ForwardingHandler[" + getActionDescription() + "]";
    }

    /**
     * Forwards an issue or activity to another agent.
     */
    class ForwardingAction extends HandlerAction {

	String toName;
	AgendaItem item;
	VerbCapability requiredCapability;

	ForwardingAction(String toName, AgendaItem item) {
	    this.toName = toName;
	    this.shortDescription = action + " to " + toName;
	    this.item = item;
	    this.requiredCapability = VerbCapability.from(item.getPattern());
	}
	
	public void handle(AgendaItem item) {
	    Debug.expect(item == this.item);
	    item.getAbout().forwardTo(toName, reportBack);
	    item.setStatus(reportBack ? Status.EXECUTING : Status.COMPLETE);
        }

	public boolean isStillValid() {
	    AgentData data = contactManager.getAgentData(toName);
	    return data != null
		&& data.getRelationship() == rel
		&& data.hasCapability(requiredCapability, capabilityDefault);
	}

    }

}

