/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Oct  2 16:40:38 2006 by Jeff Dalton
 * Copyright: (c) 2001, 2004, 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import ix.icore.Status;

import ix.util.*;

/**
 * Generates HandlerActions that let a user do nothing or
 * make an {@link AgendaItem} complete.
 */
public class CompletionHandler extends SimpleCompletionHandler {

    Ip2 ip2;

    public CompletionHandler(Ip2 ip2) {
	super();
	this.ip2 = ip2;
    }

    public void addHandlerActions(AgendaItem item) {
	// Standard options that all items have.
	// The ActivityAgenda.OtherReportsActivity wantsActionsFrom
	// method must allow actions from this class.  /\/
	item.addAction(new NoAction(item));
	item.addAction(new DoneAction(item));
	item.addAction(new NotApplicableAction(item));
    }

    public static class NoAction extends HandlerAction.NoAction {

	// /\/: The AgendaItem is needed because getActionDescription
	// may need to call makeDescription.
	AgendaItem item;

	public NoAction(AgendaItem item) {
	    // this.shortDescription = "No Action";
	    this.shortDescription = makeDescription(item);
	    this.item = item;
	}

	// /\/: Is this ever called any more [02 Oct 06]?
	// It will probably confuse a lot of things (such as logging)
	// if there are "handlings" that aren't supposed to count.
	public void handle(AgendaItem item) {
	    // N.B. no status change.
	    Debug.noteln("No action for", item);
	    // Have to clear the handledBy field.
	    Debug.expect(item.getHandledBy() == this);
	    item.setHandledBy(null);
	}

	public String getActionDescription() {
	    // /\/: If we're reloading viewers, the current description
	    // may not be right.
	    return shortDescription = makeDescription(item);
	}

	String makeDescription(AgendaItem item) {
	    if (item instanceof ActivityItem) {
		ActivityItem ai = (ActivityItem)item;
		String rName = ai.getExpansionRefinementName();
		if (rName != null)
		    // return "Expanded by " + rName;
                    return "Refined by " + rName;
	    }
	    String how = item.getHowHandled();
	    if (how != null)
		return how;
            if (item.getStatus() == Status.COMPLETE)
                // /\/: We'll say "Done" if the item is COMPLETE
                // though in some cases it should be "N/A" instead.
                // Perhaps we should always remember how the item
                // was handled, but would that always make sense
                // when a plan from one kind of agent is loaded in another?
                return "Done";
            else
                return NO_ACTION_DESCRIPTION;
	}

    }

    public class DoneAction extends HandlerAction.Manual {

	AgendaItem item;

	public DoneAction(AgendaItem item) {
	    this(item, "Done");
	}

	public DoneAction(AgendaItem item, String descr) {
	    this.item = item;
	    this.shortDescription = descr;
	}

	public boolean isReady() {
	    return Collect.isEmpty
		(ip2.getModelManager().getNodeConditions(item));
	}

	public ActionUnreadyReason getUnreadyReason() {
	    return new SimpleUnreadyReason(this, new String[] {
		"Cannot be used when there are conditions.",
		"Use \"Satisfy conditions\" instead."});
	}

    }

    public class NotApplicableAction extends DoneAction {
	public NotApplicableAction(AgendaItem item) {
	    super(item, "N/A");
	}
    }

}
