/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Oct  2 17:01:38 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2004, 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.iface.util.Reporting;

import ix.ip2.event.*;
import ix.ip2.log.ItemHandledEvent;

import ix.icore.*;
import ix.icore.process.PNode;
import ix.icore.domain.Refinement;
import ix.icore.log.HistoryEvent;

import ix.util.*;
import ix.util.lisp.*;
import ix.util.match.MatchEnv;
import ix.util.context.*;

/**
 * An entry in an Agenda.
 */
public abstract class AgendaItem extends PNode {

    // /\/: This is context-dependent (only) so that we
    // change it for Slip goal nodes.  But that matters only
    // when we're off in the planner, and so we can normally
    // but the value in the root context so that its visible
    // everywhere.  [This makes things easier for the ItemEditor
    // when it is editing an item and the option changes and
    // the user asks for a copy or a view of the item.]
    private ContextValue __about;

    // /\/: Comments are now an annotation on the TaskItem
    // protected String comments = "";

    protected List actions = new LinkedList();
    protected List reports = new LinkedList();
    protected List listeners = new LinkedList();

    protected HandlerAction handledBy = null;

    public AgendaItem(TaskItem about) {
	this(null, about);
    }

    public AgendaItem(AgendaItem parent, TaskItem about) {
	super(parent);
	// Put the initial "about" in the root context.
	__about = new TypedContextValue(TaskItem.class, about);
	if (parent != null && !about.priorityWasSet())
	    about.setPriority(parent.getPriority());
	// /\/: Don't make any status decisions until the item
	// is added to the Agenda.  This is so that ActivityItems
	// can consult the model manager.
	// computeStatus();
    }

    public TaskItem getAbout() {
	return (TaskItem)__about.get();
    }

    public void setAbout(TaskItem item) {
	// /\/: This is needed only for Slip goal nodes.
	__about.set(item);
    }

    public LList getPattern() {
	return getAbout().getPattern();
    }

    public Set getPatternVars() {
	return getAbout().getPatternVars();
    }

    public Set getUnboundVars() {
	// Start with the ones from the issue or activity -
	// the vars in the pattern.
	Set vars = getAbout().getUnboundVars();

	// Add unbound vars from this item's varTable -
	// from the top level of expansion.
	if (getVarTable() != null)
	    vars.addAll(Variable.unboundVarsIn(getVarTable().values()));

	// /\/: How about from all descendent nodes?

	return vars;
    }

    public void setPattern(LList pattern) {
	getAbout().setPattern(pattern);
    }

    public String getShortDescription() {
	// N.B. changes when variables become bound, but since we
	// recompute it every time, we should always have the right
	// value.
	return PatternParser.unparse(getPattern());
    }

    public boolean isNew() {
        return getAbout().getAnnotation("is new") != null;
    }

    public void setIsNew(boolean value) {
        Debug.noteln("setIsNew of " + this + " = " + value);
        if (value)
            getAbout().setAnnotation("is new", Boolean.TRUE);
        else
            getAbout().removeAnnotation("is new");
        fireAgendaItemEdited();
    }

    public AgendaItem getParent() {
	return (AgendaItem)parent;
    }

    public Status getStatus() {
	return getAbout().getStatus();
    }
    public void setStatus(Status status) {
	if (this.getStatus() != status) {
	    if (getAbout().getReportBack() == YesNo.YES)
                // Status isn't changed until after any report-back
                // in case we can't send the report.  But is that right? /\/
		handleReportBack(status);
            Ip2ModelManager mm = (Ip2ModelManager)getModelManager();
            // /\/: In ix.ip2.ActivityAgenda.addItemsBefore,
            // at least when called by ix.iplan.SlipAchieveConds.run,
            // the MM is null; but we don't want to undo then anyway.
            if (mm != null)
                mm.saveUndoAction(new UndoSetStatus());
	    getAbout().setStatus(status);
	    super.setStatus(status);
	    fireStatusChanged();
	}
    }

    public void assignStatus(Status s) {
	getAbout().setStatus(s);
	super.assignStatus(s);
	fireStatusChanged();	// ?? /\/
    }

    class UndoSetStatus extends AbstractUndoAction {
        Status saved = getAbout().getStatus();
        UndoSetStatus() {
            super("set status");
        }
        public void undo() {
            getAbout().setStatus(saved);
        }
    }

    public Priority getPriority() {
	return getAbout().getPriority();
    }
    public void setPriority(Priority priority) {
	Debug.noteln("Changing priority of " + this + " to " + priority);
        ((Ip2ModelManager)getModelManager())
            .saveUndoAction(new UndoSetPriority());
	getAbout().setPriority(priority);
	firePriorityChanged();
    }

    class UndoSetPriority extends AbstractUndoAction {
        Priority saved = getAbout().getPriority();
        UndoSetPriority() {
            super("set priority");
        }
        public void undo() {
           getAbout().setPriority(saved);
        }
    }

    public String getComments() {
	String comments = getAbout().getComments();
	return comments == null ? "" : comments;
    }
    public void setComments(String comments) {
        Ip2ModelManager mm = (Ip2ModelManager)getModelManager();
        // /\/: If we are doing this before the item has been
        // added to the model, the model-manager will be null
        // and we also shouldn't make an undo record.
        // Some of the other "set" methods might eventually
        // have to make a similar test.
        if (mm != null)
            mm.saveUndoAction(new UndoSetComments());
	getAbout().setComments(comments);
	fireAgendaItemEdited();
    }

    class UndoSetComments extends AbstractUndoAction {
        String saved = getAbout().getComments(); // may be null
        UndoSetComments() {
            super("set comments");
        }
        public void undo() {
            getAbout().setComments(saved);
            // /\/: Why don't we need to fire anything in other
            // undo cases?  probably because we reload the viewers.
            fireAgendaItemEdited();
        }
    }

    public List getActions() {
	return actions;
    }

    public void clearActions() {
	actions.clear();
    }

    /**
     * Gives this item its say in whether an item-handler should
     * be able to give it handler actions.
     *
     * @see ItemHandler#appliesTo(AgendaItem item)
     */
    public boolean wantsActionsFrom(ItemHandler handler) {
	return true;
    }

    /**
     * Add an action to this item.  Note that an action with the same
     * description may already exist.  We regard that as a problem for
     * the AgendaItemListeners.  For example, a user interface might use
     * the description as a way to identify the action and may not want
     * any ambiguity.
     */
    public void addAction(HandlerAction act) {
	Debug.noteln(this + " adding " + act);
	Debug.expect(!hasAction(act), "Action added twice", act);
	HandlerAction exists = findAction(act.getActionDescription());
	if (exists != null) {
	    Debug.noteln("Action description not unique",
			 Util.quote(act.getActionDescription()));
	}
	actions.add(act);
	act.computeStatus();
	fireNewHandlerAction(act);
    }

    /**
     * Adds an action to this item, putting it at a "nice" place
     * in the list.  This is typically used by the
     * {@link ItemHandler#reviseHandlerActions(AgendaItem, Object)}
     * method.
     */
    public void insertAction(HandlerAction action) {
	Debug.noteln(this + " inserting " + action);
	Debug.expect(!hasAction(action), "Action added twice", action);
	String description = action.getActionDescription();
	String word1 = Strings.beforeFirst(" ", description);
	boolean sawWord1 = false;
	boolean added = false;
        word1 = word1 + " ";    // avoid partial matches
	for (ListIterator i = actions.listIterator(); i.hasNext();) {
	    HandlerAction act = (HandlerAction)i.next();
	    String descr = act.getActionDescription();
	    if (descr.equals(description)) {
		Debug.noteln("Action description not unique",
			     Strings.quote(descr));
		sawWord1 = true;
	    }
	    else if (descr.startsWith(word1)) {
		sawWord1 = true;
	    }
	    else if (sawWord1) {
		// If we've seen word1, add before the next action
		// that doesn't begin with the same word.
		i.previous();
		i.add(action);
		Debug.expect(i.next() == act);
		added = true;
		break;
	    }
	}
	if (!added) actions.add(action);
	action.computeStatus();
	fireNewHandlerAction(action);
    }

    public boolean hasAction(HandlerAction act) {
	return actions.contains(act);
    }

    public HandlerAction findAction(Class actionClass) {
	// N.B. returns the 1st that matches; there may be > 1.
	return (HandlerAction)
	    Collect.findIf(actions, Fn.isInstanceOf(actionClass));
    }

    public List findAllActions(Class actionClass) {
	return (List)Collect.filter(actions, Fn.isInstanceOf(actionClass));
    }

    public HandlerAction findAction(String actionDescription) {
	// N.B. returns the 1st that matches; there may be > 1.
	for (Iterator i = actions.iterator(); i.hasNext();) {
	    HandlerAction act = (HandlerAction)i.next();
	    if (act.getActionDescription().equals(actionDescription))
		return act;
	}
	return null;
    }

    public boolean actionCanBeTakenNow(HandlerAction act) {
	// Normally, allow any action when status is POSSIBLE and
	// no action when the status is something else.
	// /\/: Not clear what we should do when the usuer manually
	// specifies an expansion.  Right now, this method isn't called.
	return act.canAlwaysBeTakenNow()
	    || (getStatus() == Status.POSSIBLE && act.isReady());
    }

    public HandlerAction getHandledBy() {
	return handledBy;
    }

    public void setHandledBy(HandlerAction act) {
	handledBy = act;
    }

    public String getHowHandled() {
	// /\/: Why this -and- getHandledBy()?
	if (getHistory() != null) {
	    for (Iterator i = getHistory().iterator(); i.hasNext();) {
		HistoryEvent e = (HistoryEvent)i.next();
		if (e instanceof ItemHandledEvent) {
		    String description = ((ItemHandledEvent)e).getAction();
		    if (description != null)
			return description;
		}
	    }
	}
	return null;
    }

    public List getHistory() {
	return getAbout().getHistory();
    }

    public void addHistoryEvent(HistoryEvent event) {
	Ip2ModelManager mm = (Ip2ModelManager)getModelManager();
	if (mm != null)
	    mm.saveUndoAction(new UndoAddHistoryEvent(event));
	getAbout().addHistoryEvent(event);
    }

    class UndoAddHistoryEvent extends AbstractUndoAction {
	HistoryEvent event;
	UndoAddHistoryEvent(HistoryEvent event) {
	    super("add history event");
	    this.event = event;
	}
	public void undo() {
	    List history = getHistory();
	    if (!history.remove(event))
		Debug.noteln("History did not contain", event);
	    if (history.isEmpty())
		getAbout().setHistory(null);
	}
    }

    public boolean wantsReport(Report report) {
	Name ref = report.getRef();
	return ref != null && ref.equals(getAbout().getId());
    }

    public void addReport(Report report) {
	Debug.expect(wantsReport(report), "unwanted report", report);
	Debug.noteln(this + " accepts " + report);

	String lineSeparator = System.getProperty("line.separator");
	String comments = getComments();
	comments = comments
	    + (comments.equals("") || comments.endsWith(lineSeparator)
	       ? "" : lineSeparator)
	    + Reporting.reportDescription(report) + lineSeparator;
	setComments(comments);

	reports.add(report);
	fireNewReport(report);

	setStatusBasedOn(report);

    }

    protected void setStatusBasedOn(Report report) {
	// /\/: Change status even though officially we may not
	// be able to make this transition (e.g. from COMPLETE
	// to EXECUTING).
	Status status = getStatus();
	if (status == Status.IMPOSSIBLE)
	    return;
	if (report.isCompletion()) {
	    if (report.isSuccess())
		setStatus(Status.COMPLETE);
	    else {
		Debug.expect(report.isFailure());
		setStatus(Status.IMPOSSIBLE);
	    }
	}
	else if (report.isProgress()) {
	    setStatus(Status.EXECUTING);
	}
    }

    public List getReports() {
	return reports;
    }

    protected void handleReportBack(Status newStatus) {
	Debug.expect(getAbout().getReportBack() == YesNo.YES);

	// See what we need to say
	ReportType type;
	if (newStatus == Status.COMPLETE)
	    type = ReportType.SUCCESS;
	else if (newStatus == Status.IMPOSSIBLE)
	    type = ReportType.FAILURE;
	else return;

	// Construct the Report.
	Report report = new Report("done - " + type);
	report.setReportType(type);
	report.setSenderId(Name.valueOf(IXAgent.getAgent().getAgentIPCName()));
	report.setRef(getAbout().getRef());

	// Send.
	IPC.sendObject(getAbout().getSenderId() // whoever sent this item to us
		            .toString(),   // /\/ 'cause it's a Name
		       report);
    }

    public void addItemListener(AgendaItemListener listener) {
	// /\/: What is the Java convention on adding if already there?
	if (!listeners.contains(listener))
	    listeners.add(listener);
    }

    public void fireStatusChanged() {
	AgendaItemEvent event = new AgendaItemEvent(this);
	// This method may be called before listeners has been
	// initialized because PNode init calls computeStatus.
	if (listeners != null) {
	    for (Iterator i = listeners.iterator(); i.hasNext();) {
		AgendaItemListener listener = (AgendaItemListener)i.next();
		listener.statusChanged(event);
	    }
	}
    }

    public void firePriorityChanged() {
	AgendaItemEvent event = new AgendaItemEvent(this);
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    AgendaItemListener listener = (AgendaItemListener)i.next();
	    listener.priorityChanged(event);
	}
    }

    public void fireHandlerActionsChanged() {
	AgendaItemEvent event = new AgendaItemEvent(this);
	// Debug.noteln("fireHandlerActionsChanged", event);
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    AgendaItemListener listener = (AgendaItemListener)i.next();
	    listener.handlerActionsChanged(event);
	}
    }

    public void fireNewHandlerAction(HandlerAction act) {
	AgendaItemEvent event = new AgendaItemEvent(this);
	// Debug.noteln("fireNewHandlerAction", event);
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    AgendaItemListener listener = (AgendaItemListener)i.next();
	    listener.newHandlerAction(event, act);
	}
    }

    public void fireNewReport(Report report) {
	AgendaItemEvent event = new AgendaItemEvent(this);
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    AgendaItemListener listener = (AgendaItemListener)i.next();
	    listener.newReport(event, report);
	}
    }

    public void fireAgendaItemEdited() {
	AgendaItemEvent event = new AgendaItemEvent(this);
	Debug.noteln("fireAgendaItemEdited", event);
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    AgendaItemListener listener = (AgendaItemListener)i.next();
	    listener.agendaItemEdited(event);
	}
    }

    public String toString() {
	// /\/: This can be called before __about is initialized
	// by debugging output produced during the invocation
	// of our superclass's constructor.
	if (__about == null)
	    return "Item[about <uninitialized>]";
	else
	    return "Item[" + getAbout() + "]";
    }

}

// Issues:
// * Should reports be recorded in the "about" object instead?
