/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Thu Oct 19 16:30:02 2006 by Jeff Dalton
 * Copyright: (c) 2001 - 2006, AIAI, University of Edinburgh
 */

package ix.ip2;

import java.util.*;

import ix.icore.domain.Refinement;
import ix.icore.process.ProcessModelManager;
import ix.icore.process.event.*;
import ix.icore.*;
import ix.util.*;
import ix.util.lisp.*;
import ix.util.context.*;

import ix.ip2.event.*;
import ix.ip2.log.*;

/**
 * A collection of things to do.
 */
public abstract class Agenda implements AgendaManager,
					ProcessStatusListener {

    protected PanelController controller;

    protected LLQueue items = new LLQueue();
    protected List handlers = new LinkedList();
    protected List listeners = new LinkedList();

    public Agenda(PanelController controller) {
	this.controller = controller;
    }

    public void reset() {
	items.clearCompletely();
    }

    public void clear() {
	items.clear();
    }

    public List getItems() {
	return items.contents();
    }

    public AgendaItem getItem(Name id) {
	for (Iterator i = items.iterator(); i.hasNext();) {
	    AgendaItem item = (AgendaItem)i.next();
	    Name itemId = item.getAbout().getId();
	    if (itemId != null && itemId.equals(id))
		return item;
	}
	return null;
    }

    public void addItem(AgendaItem item) {
	// ... subclasses may do some things first ...
        // controller.getModelManager().markUndoPoint("Add agenda-item");
	checkForDuplicateId(item);
	addHandlerActions(item);
	items.add(item);
	fireItemAdded(item);
    }

    public void addItemsBefore(AgendaItem at, List addList) {
        controller.getModelManager().markUndoPoint("Add items before");	
        LList oldItems = items.contents();
	LList newItems =
	    (LList)Collect.insertBeforeFirst(at, oldItems, addList);
	items.setContents(newItems);
	for (Iterator i = addList.iterator(); i.hasNext();) {
	    AgendaItem item = (AgendaItem)i.next();
	    checkForDuplicateId(item);
	    addHandlerActions(item);
	    // items.add(item);
	    fireItemAdded(item);
	}
    }

    private void checkForDuplicateId(AgendaItem item) {
	TaskItem about = item.getAbout();
	Name id = about.getId();
	if (id != null && getItem(id) != null) {
	    String className =
		Strings.afterLast(".", about.getClass().getName());
	    Debug.warn("Adding " + className +
		       " with pattern " + about.getPattern() +
		       " even though there is already another " + className +
		       " with id " + id);
	}
    }

    public void removeItem(AgendaItem item) {
        controller.getModelManager().markUndoPoint("Remove item");
	items.remove(item);
	fireItemRemoved(item);
    }

    public AgendaItem makeItem(String text) {
	return makeItem(PatternParser.parse(text));
    }

    // /\/: Not really needed now that we have the
    // makeItem(TaskItem) method.
    public abstract AgendaItem makeItem(LList pattern);

    public abstract AgendaItem makeItem(TaskItem issueOrActivity);

    public boolean canHandleAutomatically(TaskItem issueOrActivity) {
	AgendaItem item = makeItem(issueOrActivity);
	for (Iterator i = handlers.iterator(); i.hasNext();) {
	    ItemHandler h = (ItemHandler)i.next();
	    if (h.isAutomatic()
		  && h.appliesTo(item)
		  && item.wantsActionsFrom(h))
		return true;
	}
	return false;
    }

    public void handleAutomatically(TaskItem issueOrActivity) {
	AgendaItem item = makeItem(issueOrActivity);
	ProcessModelManager mm = controller.getModelManager();
	item.setPattern(mm.putVariablesInPattern(item.getPattern()));
	for (Iterator i = handlers.iterator(); i.hasNext();) {
	    ItemHandler h = (ItemHandler)i.next();
	    if (!(h.isAutomatic()
		    && h.appliesTo(item)
 		    && item.wantsActionsFrom(h)))
		continue;
	    h.addHandlerActions(item);
	    for (Iterator j = item.getActions().iterator(); j.hasNext();) {
		HandlerAction act = (HandlerAction)j.next();
		if (act.isReady()) {
		    handleItemDirectly(item, act);
		    return;
		}
	    }
	    item.clearActions();
	}
	throw new IllegalStateException
	    ("Failed to automatically handle " + issueOrActivity);
    }

    public void addHandlerActions(AgendaItem item) {
	// Add actions from the handlers
	for (Iterator i = handlers.iterator(); i.hasNext();) {
	    ItemHandler h = (ItemHandler)i.next();
	    if (h.appliesTo(item) && item.wantsActionsFrom(h))
		h.addHandlerActions(item);
	}
    }

    public void handleItem(final AgendaItem item, final HandlerAction act) {
	Debug.expect(items.contains(item), "Stray item", item);
        // /\/: How should automatic handling fit with undo?
        controller.getModelManager().undoableTransaction
            ("Handle item: " + act.getActionDescription(),
             new Runnable() {
                 public void run() {
                     handleItemDirectly(item, act);
                 }
             });
    }

    protected void handleItemDirectly(AgendaItem item, HandlerAction act) {
	Debug.expect(item.hasAction(act), "Stray action", act);
	item.setHandledBy(act);
	try {
	    act.handle(item);
	}
	catch (RuntimeException e) {
	    item.setHandledBy(null);
	    throw e;
	}
	catch (Throwable t) {
	    item.setHandledBy(null);
	    Debug.noteException(t);
	    throw new RethrownException(t);
	}
	logItemHandled(item, act);
    }

    void logItemHandled(AgendaItem item, HandlerAction act) {
	// Add to the agent's log file only if we're in an
	// appropriate option.
	if (!controller.getAgent().getOptionManager().canTakeInput())
	    return;
	ItemHandledEvent event;
	if (item instanceof IssueItem)
	    event = new IssueHandledEvent();
	else
	    event = new ActivityHandledEvent();
	event.setPattern((LList)Variable.removeVars(item.getPattern()));
	event.setAction(act.getActionDescription());
	controller.getAgent().log(event);
    }

    public void addItemHandler(ItemHandler handler) {
	handlers.add(handler);
	for (Iterator i = items.iterator(); i.hasNext();) {
	    AgendaItem item = (AgendaItem)i.next();
	    if (handler.appliesTo(item) && item.wantsActionsFrom(handler))
		handler.addHandlerActions(item);
	}
    }

    public List getItemHandlers() {
	return handlers;
    }

    public List getSyntaxList() {
	LListCollector syntaxes = new LListCollector();
	for (Iterator i = handlers.iterator(); i.hasNext();) {
	    ItemHandler h = (ItemHandler)i.next();
	    // Skip automatic handlers because the user won't
	    // use them?  /\/
	    if (!h.isAutomatic())
		syntaxes.addAll(Collect.ensureList(h.getSyntaxList()));
	}
	return syntaxes.contents();
    }

    public void expandItem(AgendaItem item, Refinement instructions) {
	throw new Error("Attempt to expand " + item);
    }

    public boolean acceptReport(Report report) {
	for (Iterator i = items.iterator(); i.hasNext(); ) {
	    AgendaItem item = (AgendaItem)i.next();
	    if (item.wantsReport(report)) {
		item.addReport(report);
		return true;
	    }
	}
	return false;
    }

    public boolean wantsReport(Report report) {
	for (Iterator i = items.iterator(); i.hasNext(); ) {
	    AgendaItem item = (AgendaItem)i.next();
	    // Skip any OtherReportsActivity.
	    if (item instanceof ActivityAgenda.OtherReportsActivity)
		continue;
	    if (item.wantsReport(report))
		return true;
	}
	return false;
    }

    /*
     * Methods in the ProcessStatusListener interface
     */

    // /\/: It's not clear that this is right.  Perhaps things that
    // want to know about bindings should listen to the model manager
    // directly.  Also, it might be better to have > 1 kind of MM listener,
    // because something that wants to know about bindings, e.g., might
    // not care about world-state changes.

    public void statusUpdate(ProcessStatusEvent e) {
    }

    public void newBindings(ProcessStatusEvent e, Map bindings) {
	// Pass along the event ...
	fireNewBindings(bindings);
	// Remove handler actions that are no longer valid
	checkActionValidity();
	// Recompute the status of all the handler actions
	computeActionStatus();
    }

    public void stateChange(ProcessStatusEvent e, Map delta) {
	computeActionStatus();
    }

    public void stateDeletion(ProcessStatusEvent e, Map delta) {
	computeActionStatus();
    }

    /*
     * Methods to manage changes in handler actions.
     */

    /**
     * Called by the controller when a handler has discovered that
     * it may need to add actions.  This method should not be called
     * directly by handlers.
     */
    public void reconsiderHandler(ItemHandler handler, Object reason) {
	// /\/: See comments in PanelController.reconsiderHandler(h).
	for (Iterator i = items.iterator(); i.hasNext();) {
	    AgendaItem item = (AgendaItem)i.next();
	    if (item.getHandledBy() == null) 	// not already handled
		if (handler.appliesTo(item)) 	// ??? /\/
		    // Shouldn't we call item.wantsActionsFrom(handler)? /\/
		    // Compare addItemHandler(ItemHandler handler).
		    handler.reviseHandlerActions(item, reason);
	}
    }

    /**
     * Called by the controller when a handler has discovered that
     * some actions may need to be deleted.  This method should not
     * be called directly by handlers.
     */
    public void checkActionValidity(ItemHandler handler, Object reason) {
	checkActionValidity();
    }

    /**
     * Checks the validity of all handler-actions.
     */
    protected void checkActionValidity() {
	for (Iterator i = items.iterator(); i.hasNext();) {
	    AgendaItem item = (AgendaItem)i.next();
	    if (item.getHandledBy() == null)	// not already handled
		checkActionValidity(item);
	}
    }

    protected void checkActionValidity(AgendaItem item) {
	boolean changes = false;
	for (Iterator i = item.getActions().iterator(); i.hasNext();) {
	    HandlerAction act = (HandlerAction)i.next();
	    if (!act.isStillValid()) {
		Debug.noteln("Removing " + act + " from " + item);
		i.remove();
		changes = true;
	    }
	}
	if (changes == true)
	    item.fireHandlerActionsChanged(); // define a diff event? /\/
    }

    protected void computeActionStatus() {
	for (Iterator i = items.iterator(); i.hasNext();) {
	    AgendaItem item = (AgendaItem)i.next();
	    computeActionStatus(item);
	}
    }

    protected void computeActionStatus(AgendaItem item) {
	// /\/: Probably should be a method of the AgendaItem,
	// but we don't want to commit too much to the
	// computeStatus way of doing things.
	Status status = item.getStatus();
	if (status == Status.COMPLETE 
	       || status == Status.EXECUTING
	       || status == Status.IMPOSSIBLE)
	    return;
	boolean changes = false;
	for (Iterator i = item.getActions().iterator(); i.hasNext();) {
	    HandlerAction act = (HandlerAction)i.next();
	    boolean oldStatus = act.isReady();
	    act.computeStatus();
	    if (act.isReady() != oldStatus) {
		changes = true;
	    }
	}
	if (changes == true)
	    item.fireHandlerActionsChanged();
    }


    /*
     * AgendaListeners and related methods.
     */

    public void addAgendaListener(AgendaListener listener) {
	listeners.add(listener);
    }

    public void fireItemAdded(AgendaItem item) {
	Debug.noteln("fireItemAdded", item);
	AgendaEvent event = new AgendaEvent(this);
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    AgendaListener listener = (AgendaListener)i.next();
	    listener.itemAdded(event, item);
	}
    }

    public void fireItemRemoved(AgendaItem item) {
	Debug.noteln("fireItemRemoved", item);
	AgendaEvent event = new AgendaEvent(this);
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    AgendaListener listener = (AgendaListener)i.next();
	    listener.itemRemoved(event, item);
	}
    }

    public void fireItemHandled(AgendaItem item, HandlerAction action) {
	Debug.noteln("fireItemHandled " + item + " " +
		     action.getActionDescription());
	AgendaEvent event = new AgendaEvent(this);
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    AgendaListener listener = (AgendaListener)i.next();
	    listener.itemHandled(event, item, action);
	}
    }

    public void fireNewBindings(Map bindings) {
	AgendaEvent event = new AgendaEvent(this);
	for (Iterator i = listeners.iterator(); i.hasNext();) {
	    AgendaListener listener = (AgendaListener)i.next();
	    listener.newBindings(event, bindings);
	}
    }

}

// Issues:
// * What should be done when removing an item that has children?
//   Don't allow it?  Remove descendents as well?
