/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Oct 18 00:48:28 2004 by Jeff Dalton
 * Copyright: (c) 2004, AIAI, University of Edinburgh
 */

package ix.ip2;

import javax.swing.*;

import java.awt.Container;
import java.awt.Component;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.Color;
import java.awt.event.*;

import java.util.*;

import ix.ip2.event.*;

import ix.icore.*;
import ix.icore.process.PNode;
import ix.util.*;
import ix.util.lisp.*;

/**
 * A simple editor for entering and viewing Activities
 */
class ActivityInsertionEditor extends ActivityEditor {

    static final String BEFORE = "before";
    static final String AFTER = "after";

    Ip2 ip2;

    JLabel descriptionLabel = new JLabel("Insert ...", JLabel.CENTER);

    String placement = null;	// BEFORE or AFTER

    AgendaItem target;

    ActivityInsertionEditor(Ip2 ip2) {
	super(ip2.activityViewer, // /\/ direct ref
	      ip2.getAgentDisplayName()
	        + " Activity Insertion Editor");
	this.ip2 = ip2;
	// Put a description label centered at the top so that
	// the user can see what this editor will do.
	contentPane = frame.getContentPane();
	contentPane.add(descriptionLabel, BorderLayout.NORTH);
	// Without a border around the editPanel, the label
	// has too much space under it.  Somehow the main Ip2
	// frame has a line below the label, presumably from a
	// border somewhere.  /\/
	editPanel.setBorder(BorderFactory
			      // line only at the top
			      .createMatteBorder(1,0,0,0,Color.black));
	frame.pack();
    }

    void insertBefore(AgendaItem item) {
	setupForInsertion(BEFORE, item);
    }

    void insertAfter(AgendaItem item) {
	setupForInsertion(AFTER, item);
    }

    void setupForInsertion(String placement, AgendaItem item) {
	showNewItem();
	this.placement = placement;
	this.target = item;
	String itemText = PatternParser.unparse(item.getPattern());
	descriptionLabel.setText("Insert " + placement + " " + itemText);
    }

    protected EditPanel makeEditPanel() {
	return new InsertionEditPanel();
    }

    class InsertionEditPanel extends ActivityEditPanel {
	void addItem() {
	    Debug.expect(editingItem == null);
	    AgendaItem i = getItem();
	    if (i != null) {
		insertItem(i);
		showItem(i);	// now editing that item
	    }
	}
    }

    void insertItem(AgendaItem item) {
	if (placement == BEFORE)
	    insertBeforeTarget(item);
	else if (placement == AFTER)
	    insertAfterTarget(item);
	else
	    throw new ConsistencyException();
	ip2.resetViewers();
	ip2.reloadViewers();
    }

    void insertBeforeTarget(AgendaItem item) {
	Debug.expect(target.getStatus() != Status.EXECUTING &&
		     target.getStatus() != Status.COMPLETE);
	Debug.expect(target.getStatus() == Status.BLANK ||
		     target.getStatus() == Status.POSSIBLE);
	// We can always put the item right before the target,
	// because we're called only when the target hasn't
	// started executing.
	Agenda actAgenda = ip2.getController().getActivityAgenda();
	actAgenda.addItemsBefore(target, Lisp.list(item));
	// Link the item before the target.
	item.getEnd().linkBefore(target.getBegin());
	// The target can no longer have status POSSIBLE,
	// because there's now a node linked before it.
	target.setStatus(Status.BLANK);
	// Make the item a sibling of the target if the target
	// has a parent.
	PNode parent = target.getParent();
	if (parent != null) {
	    parent.addChild(item, target);
	}
    }

    void insertAfterTarget(AgendaItem item) {
	// We don't want to put the item right after the target
	// if making it a sibling of the target would link it
	// before a perent end that has already completed.  
	AgendaItem where = target;// insertion point
	while (where.getParent() != null
	       && where.getParent().getStatus() == Status.COMPLETE) {
	    where = where.getParent();
	}
	// Link the item after the target.  We do this before
	// adding the item to the agenda, because the link
	// affects its status.
	target.getEnd().linkBefore(item.getBegin());
	// We have an add-before operation, but not an add-after,
	// so we have to use the node listed after the insertion point.
	Agenda actAgenda = ip2.getController().getActivityAgenda();
	AgendaItem next = (AgendaItem)itemAfter(where, actAgenda.getItems());
	if (next != null)
	    actAgenda.addItemsBefore(next, Lisp.list(item));
	else // put at end
	    actAgenda.addItem(item);
	// Make the item a sibling of the insertion point
	// if the insertion point has a parent.
	PNode parent = where.getParent();
	if (parent != null) {
	    // We again have an operation that adds before but not
	    // one that adds after.
	    AgendaItem nextSibling = // it's ok for the value to be null
		(AgendaItem)itemAfter(where, parent.getChildren());
	    parent.addChild(item, nextSibling);
	    // If the end-end of the parent was POSSIBLE before,
	    // it's not now, because the new child is linked
	    // before it.
	    if (parent.getEnd().getStatus() == Status.POSSIBLE)
		parent.getEnd().setStatus(Status.BLANK);
	}
    }

    Object itemAfter(Object target, List list) {
	for (Iterator i = list.iterator(); i.hasNext();) {
	    Object item = i.next();
	    if (item == target) {
		if (i.hasNext())
		    return i.next();
		else
		    return null;
	    }
	}
	return null;
    }

}
