/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Mar 12 02:52:09 2006 by Jeff Dalton
 * Copyright: (c) 2001, 2005, 2006, AIAI, University of Edinburgh
 */

package ix.iface.util;

import java.awt.Font;
import java.util.*;
import javax.swing.*;

import ix.util.*;

/**
 * A JTextArea for recording transcripts and similar information.
 */
public class TranscriptTextArea extends JTextArea {

    protected boolean foldLongLines = false;
    int excessLengthAllowed = 0;

    /**
     * Constructs an empty text area of the specified size.
     */
    public TranscriptTextArea(int rows, int cols) {
	super(rows, cols);
    }

    /**
     * Tells {@link #appendLine(String)} to break a long line
     * into separate lines that will fit in the display area.
     * It Also causes a fixed-width font to be used.
     *
     * @param excessAllowed  the number of extra characters allowed
     *    before a line must be folded.
     */
    public void setFoldLongLines(int excessAllowed) {
	this.foldLongLines = true;
	this.excessLengthAllowed = excessAllowed;
	// We want a fixed-width font
	Debug.noteln("Have font", getFont());
	Font font = new Font("Monospaced", Font.PLAIN, 11);
	setFont(font);
    }

    /**
     * Appends the text, then moves the caret position to the end,
     * which should also cause scrolling to the end.  Note that the
     * other append methods directly or indirectly call this one.
     */
    public void append(String text) {
	super.append(text);
	// Setting caret position to end should scoll to the end
	setCaretPosition(getText().length());
    }

    /**
     * Appends the text, followed by a line separator.
     * The line separator is the value of
     * <tt>System.getProperty("line.separator")</tt>.
     */
    public void appendLine(String text) {
	String lineSeparator = System.getProperty("line.separator");
	if (foldLongLines) {
	    // /\/: The number of columns doesn't seem to change
	    // if the user stretches the containing window.
  	    // So can't just do getColumns() + excessLengthAllowed.
	    double width = getEffectiveWidth();
	    double charWidth = width / getColumnWidth();
	    charWidth -= 3;	// estimated scrollbar width
	    int maxLen = (int)charWidth + excessLengthAllowed;
	    Debug.noteln("maxLen =", maxLen);
	    if (text.length() > maxLen) {
		// This line might secretly contain newlines,
		// but that's handled by Strings.foldLongLines.
		append(Strings.foldLongLine(text, maxLen, "   ")
		       + lineSeparator);
		return;
	    }
	}
	append(text + lineSeparator);
    }

    protected double getEffectiveWidth() {
	return getSize().getWidth();
    }

    /**
     * Breaks the text into lines, then appends the lines, each
     * prefixed by the specified number of spaces.
     */
    public void appendIndentedLines(int indent, String text) {
	List lines = Strings.breakIntoLines(text);
	for (Iterator i = lines.iterator(); i.hasNext();) {
	    String line = (String)i.next();
	    appendLine(Strings.repeat(indent, " ") + line);
	}
    }

}

