/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Sun Jul 20 20:44:54 2003 by Jeff Dalton
 * Copyright: (c) 2003, AIAI, University of Edinburgh
 */

package ix.iface.util;

import java.awt.event.*;
import javax.swing.JMenuItem;
import java.util.*;

import ix.util.*;

/**
 * Manages the "tools" for an I-X agent's user interface.
 * Also provides and manages a JMenu sutiable for use in a menu-bar.
 *
 * @see ToolController
 */
public class ToolManager {

    protected List controllers = new LinkedList();

    protected AlphabetizingMenu toolsMenu = new AlphabetizingMenu("Tools");

    public ToolManager() {
    }

    /**
     * Returns a JMenu suitable for use in a menu bar.  New entries
     * are normally added to the menu only by calling the ToolManager's
     * {@link #addTool(ToolController)} method.  In that way, the
     * menu entry will automatically be given an ActionListener
     * that calls {@link #activateTool(ToolController)}.
     */
    public synchronized AlphabetizingMenu getToolsMenu() {
	return toolsMenu;
    }

    /**
     * Returns the {@link ToolController} for the tool with the
     * specified name, or null if no such tool is known to this
     * ToolManager.
     */
    public synchronized ToolController findTool(String toolName) {
	for (Iterator i = controllers.iterator(); i.hasNext();) {
	    ToolController tc = (ToolController)i.next();
	    if (tc.getToolName().equals(toolName))
		return tc;
	}
	return null;
    }

    /**
     * Calls {@link #findTool(String)} to obtain the {@link ToolController}
     * for the tool with the specified name.  If that result is non-null,
     * it is returned; otherwise an exception is thrown.
     *
     * @throws IllegalArgumentException if no tool of the specified
     *    name is known to this ToolManager.
     */
    public ToolController findToolElseError(String toolName) {
	ToolController tc = findTool(toolName);
	if (tc != null)
	    return tc;
	else
	    throw new IllegalArgumentException
		("There is no tool named " + Strings.quote(toolName));
    }

    /**
     * Adds a tool to the list of tools managed by this ToolManager.
     * An entry is also added to the menu that would be returned by
     * the {@link #getToolsMenu()} method.  The menu entry is given
     * an ActionListener that calls {@link #activateTool(ToolController)}
     * on the tool's controller.
     *
     * @see #makeMenuItem(ToolController)
     */
    public synchronized void addTool(ToolController tc) {
	if (findTool(tc.getToolName()) != null)
	    throw new IllegalArgumentException
		("Attempt to add a second tool named " +
		 Strings.quote(tc.getToolName()));
	controllers.add(tc);
	toolsMenu.insert(makeMenuItem(tc));
    }

    /**
     * Makes a JMenuItem that has an ActionListener that calls
     * {@link #activateTool(ToolController)} on the specified
     * controller.  The text of the menu entry is obtained by
     * calling {@link ToolController#getToolName()}.
     */
    protected JMenuItem makeMenuItem(final ToolController tc) {
	JMenuItem item = new JMenuItem(tc.getToolName());
	item.addActionListener(new CatchingActionListener() {
	    protected void innerActionPerformed(ActionEvent event) {
		Debug.expect(event.getActionCommand()
			     .equals(tc.getToolName()));
		activateTool(tc);
	    }
	});
	return item;
    }

    /**
     * Creates the tool if it does not already exist and then
     * makes it visible.  This is done by calling
     * {@link ToolController#ensureToolVisible()}.
     */
    protected void activateTool(ToolController tc) {
	tc.ensureToolVisible();
    }

}
