/* Author: Jeff Dalton <J.Dalton@ed.ac.uk>
 * Updated: Mon Oct 25 14:41:09 2004 by Jeff Dalton
 * Copyright: (c) 2002, 2003, AIAI, University of Edinburgh
 */

package ix.iface.util;

import java.lang.reflect.Method;
import java.util.*;

import ix.util.*;

/**
 * An object used to manage a "tool" -- a utility usually reached
 * via the "Tools" menu of an I-X agent's user interface.
 *
 * <p>ToolController provides reasonable defaults for most methods.
 * Normally, a subclass needs to define only the {@link #createTool()}
 * method.  However, if the tool object lacks a setVisible(boolean)
 * method, {@link #setToolVisible(boolean)} must also be defined.
 * A typical use is therefore something like this:
 * <pre>
 *    ToolManager toolManager = new ToolManager();
 *    ...
 *    toolManager.addTool(new ToolController("XML Editor") {
 *        public Object createTool() {
 *            return new XMLTreeEditorFrame(...);
 *        }
 *    });
 * </pre>
 * </p>
 *
 * @see ToolManager
 */
public abstract class ToolController {

    protected String toolName;
    protected Object tool = null;

    protected ToolController() {}

    public ToolController(String toolName) {
	this.toolName = toolName;
    }

    /**
     * Returns a string that can be used to identify the tool in a menu
     * or in other lookup mechanisms.
     */
    public String getToolName() {
	return toolName;
    }

    /**
     * Consturucts the tool when it does not already exist.
     *
     * <p>This method should be called only if the {@link #getTool()}
     * method returns <code>null</code>, and it should not normally
     * be called directly.  Call {@link #setTool()} instead, or use
     * a method that calls {@link #setTool()}, such as 
     * {@link #ensureTool()}.</p>
     *
     * <p>That ensures that createTool is called only once per
     * ToolController and that the controller remembers the tool
     * so that {@link #getTool()} can return it.   Note, however,
     * that it is not strictly necessary for createTool to construct
     * a new object.  In some cases, it might return an existing
     * object that has not yet been made known to this ToolController.
     * For example, if two different frames share a tool, they would
     * have separate {@link ToolManager}s and separate ToolControllers
     * for the same tool object, but at most one of those controllers
     * should construct a new tool object.</p>
     */
    protected abstract Object createTool();

    /**
     * Returns the tool object if it exists, otherwise <code>null</code>.
     */
    public Object getTool() {
	return tool;
    }

    /**
     * Returns the tool after calling {@link #setTool()} if the
     * tool has not already been created.
     */
    public Object ensureTool() {
	if (tool == null)
	    setTool();
	return tool;
    }

    /**
     * Returns the tool after calling {@link #ensureTool()}
     * and {@link #setToolVisible(boolean)} with argument
     * <code>true</code>.
     */
    public Object ensureToolVisible() {
	ensureTool();
	setToolVisible(true);
	return tool;
    }

    /**
     * Calls {@link #createTool()} and records the result so that
     * it will be returned by subsequent calls to {@link #getTool()}.
     */
    public void setTool() {
	tool = createTool();
	Debug.noteln("Assigned tool " + Strings.quote(toolName) + ": " + tool);
    }

    /**
     * Causes the tool to be visible or not.  For example, if the
     * tool object is or contains a frame, this method might call
     * the <code>setVisible</code> method of that frame.
     *
     * <p>The method provided by the ToolController class uses
     * reflection to call the tool object's setVisible(boolean)
     * method, if it has one.  If it doesn't, an UnsupportedOperationException
     * is thrown.</p>
     */
    public void setToolVisible(boolean t) {
	Class[] sig = new Class[] {Boolean.TYPE};
	Method m = Fn.getMethod(tool, "setVisible", sig);
	if (m != null) {
	    Fn.apply(tool, m, new Object[] {new Boolean(t)});
	}
	else
	    throw new UnsupportedOperationException
		("Can't make tool " + Strings.quote(toolName) +
		 " visible.");
    }

}
